package ru.yandex.direct.web.entity.uac.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiParam
import org.springframework.http.HttpStatus
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.entity.uac.service.trackingurl.ParserType.IMPRESSION_URL
import ru.yandex.direct.core.entity.uac.service.trackingurl.ParserType.TRACKING_URL
import ru.yandex.direct.core.entity.uac.service.trackingurl.TrackingUrlParseService
import ru.yandex.direct.core.entity.uac.validation.appInfoNotFound
import ru.yandex.direct.core.entity.uac.validation.invalidTrackingUrl
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.web.entity.uac.model.CreateImpressionUrlRequest
import ru.yandex.direct.web.entity.uac.model.CreatedImpressionUrl
import ru.yandex.direct.web.entity.uac.model.ValidateTrackingUrlRequest
import ru.yandex.direct.web.entity.uac.model.ValidatedTrackingUrl
import ru.yandex.direct.web.entity.uac.service.UacMobileAppService
import ru.yandex.direct.web.entity.uac.service.UacTrackingUrlStatService
import ru.yandex.direct.web.entity.uac.service.UacTrackingUrlValidationService
import ru.yandex.direct.web.entity.uac.toResponse
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService

@RestController
@Api(tags = ["uac"])
@RequestMapping("/uac", produces = [MediaType.APPLICATION_JSON_VALUE])
class UacTrackingUrlController(
    private val validationResultConversionService: ValidationResultConversionService,
    private val uacTrackingUrlValidationService: UacTrackingUrlValidationService,
    private val trackingUrlParseService: TrackingUrlParseService,
    private val uacMobileAppService: UacMobileAppService,
    private val uacTrackingUrlStatService: UacTrackingUrlStatService
) {
    companion object {
        private const val DEFAULT_DAYS_NUMBER = 90
    }

    @ApiOperation(
        value = "validateTrackingUrl",
        httpMethod = "POST",
        nickname = "validateTrackingUrl"
    )
    @PostMapping(
        value = ["/tracking_url/validate"],
        consumes = [MediaType.APPLICATION_JSON_UTF8_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    fun validateTrackingUrl(
        @RequestBody validateTrackingUrlRequest: ValidateTrackingUrlRequest
    ): ResponseEntity<Any> {

        val (defect, trackingUrl) = uacTrackingUrlValidationService.validateAndParseTrackingUrl(
            validateTrackingUrlRequest
        )
        if (defect != null) {
            return badRequestResponse(ValidationResult.failed(validateTrackingUrlRequest.url, defect))
        }
        return ResponseEntity(
            trackingUrl?.let {ValidatedTrackingUrl.fromTrackingUrl(it)}.toResponse(),
            HttpStatus.OK
        )
    }

    @ApiOperation(
        value = "createImpressionUrl",
        httpMethod = "POST",
        nickname = "createImpressionUrl"
    )
    @PostMapping(
        value = ["/tracking_url/create_impression_url"],
        consumes = [MediaType.APPLICATION_JSON_UTF8_VALUE],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    fun createImpressionUrl(
        @RequestBody createImpressionUrlRequest: CreateImpressionUrlRequest
    ): ResponseEntity<Any> {

        val (defect, impressionUrl) = trackingUrlParseService.createImpressionUrl(
            createImpressionUrlRequest.url, createImpressionUrlRequest.platform
        )
        if (defect != null) {
            return badRequestResponse(ValidationResult.failed(createImpressionUrlRequest.url, defect))
        }
        return ResponseEntity(
            impressionUrl?.let {CreatedImpressionUrl.fromTrackingUrl(it)}.toResponse(),
            HttpStatus.OK
        )
    }

    @ApiOperation(
        value = "getTrackingUrlStat",
        httpMethod = "GET",
        nickname = "getTrackingUrlStat"
    )
    @GetMapping(
        value = ["/tracking_url/stat"],
        produces = [MediaType.APPLICATION_JSON_UTF8_VALUE],
    )
    fun getTrackingUrlStat(
        @ApiParam(value = "url", required = true) @RequestParam("url") url: String,
        @ApiParam(value = "app_info_id", required = true) @RequestParam("app_info_id") appInfoId: String,
        @ApiParam(value = "days", required = false) @RequestParam("days") days: Int?
    ): ResponseEntity<Any> {

        val trackingUrl = trackingUrlParseService.parse(url, null, TRACKING_URL)
            ?: trackingUrlParseService.parse(url, null, IMPRESSION_URL)
            ?: return badRequestResponse(ValidationResult.failed(url, invalidTrackingUrl()))

        val appInfo = uacMobileAppService.getAppInfo(appInfoId) ?: return badRequestResponse(
            ValidationResult.failed(appInfoId, appInfoNotFound())
        )

        val aggregatedTrackingUrlStat = uacTrackingUrlStatService.getAggregatedStat(
            trackingUrl,
            appInfo,
            days ?: DEFAULT_DAYS_NUMBER
        )

        return ResponseEntity(
            aggregatedTrackingUrlStat.toResponse(),
            HttpStatus.OK
        )
    }

    private fun badRequestResponse(validationResult: ValidationResult<*, Defect<*>>): ResponseEntity<Any> {
        return ResponseEntity(
            validationResultConversionService.buildValidationResponse(validationResult), HttpStatus.BAD_REQUEST
        )
    }
}
