package ru.yandex.direct.web.entity.uac.converter

import java.math.BigDecimal
import java.time.LocalDateTime
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagData
import ru.yandex.direct.core.entity.uac.converter.UacContentConverter
import ru.yandex.direct.core.entity.uac.model.CampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.campaign_content.AssetStat
import ru.yandex.direct.core.entity.uac.model.campaign_content.ContentEfficiency
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.web.entity.uac.model.CreativeStatistics
import ru.yandex.direct.web.entity.uac.model.EMPTY_STAT
import ru.yandex.direct.web.entity.uac.model.UacContentStatistics
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils.fromMicros

object UacContentStatisticsConverter {

    fun toContentStatistics(
        uacCampaignContent: UacYdbCampaignContent,
        content: Content?,
        efficiency: ContentEfficiency? = null,
        partialExistence: Boolean? = null,
        creativeStatistics: CreativeStatistics? = null,
        rejectReasons: List<ModerationDiagData>? = null,
        changeRejectedAssetToCreatedBySecondsShift: Int = 0,
        bannerId: String? = null,
        bid: String? = null,
    ) = UacContentStatistics(
        id = uacCampaignContent.id,
        content = content,
        rejectReasons = rejectReasons,
        text = uacCampaignContent.text,
        contentType = uacCampaignContent.type!!,
        status = getCampaignContentStatus(uacCampaignContent, changeRejectedAssetToCreatedBySecondsShift),
        efficiency = efficiency,
        startedAt = uacCampaignContent.createdAt.atZone(DateTimeUtils.MSK).toEpochSecond(),
        deletedAt = uacCampaignContent.removedAt?.atZone(DateTimeUtils.MSK)?.toEpochSecond(),
        partialExistence = partialExistence,
        sitelink = if (uacCampaignContent.type == MediaType.SITELINK)
            UacContentConverter.toSitelink(uacCampaignContent) else null,
        creativeStatistics = creativeStatistics,
        bannerId = bannerId,
        bid = bid,
    )

    /**
     * Иногда, при создании кампании, ассеты сохраняются со статусом rejected и спустя
     * нескольких секунд переходят в статус created. Найти быстро место где это происходит не удалось,
     * поэтому добавлен этот временный костыль, пока ищем первопричину
     */
    private fun getCampaignContentStatus(
        uacCampaignContent: UacYdbCampaignContent,
        secondsShift: Int,
    ): CampaignContentStatus {
        if (secondsShift < 1 || CampaignContentStatus.REJECTED != uacCampaignContent.status) {
            return uacCampaignContent.status
        }
        val startedAtWithShift = uacCampaignContent.createdAt.plusSeconds(secondsShift.toLong())

        return if (startedAtWithShift > LocalDateTime.now()) CampaignContentStatus.CREATED
        else uacCampaignContent.status
    }

    fun AssetStat?.toCreativeStatistics(): CreativeStatistics {
        if (this == null) {
            return EMPTY_STAT
        }

        return CreativeStatistics(
            shows = shows.toLong(),
            clicks = clicks.toLong(),
            conversions = conversions.toLong(),
            costMicros = cost.toLong(),
            costCurMicros = costCur.toLong(),
            costTaxFreeMicros = costTaxFree.toLong(),
            installs = installs.toLong(),
            postViewConversions = postViewConversions.toLong(),
            postViewInstalls = postViewInstalls.toLong(),
            avgCpm = 0.0,
            avgCpc = 0.0,
            videoAvgTrueViewCost = 0.0,
            uniqViewers = 0,
            videoFirstQuartileRate = 0.0,
            videoMidpointRate = 0.0,
            videoThirdQuartileRate = 0.0,
            videoCompleteRate = 0.0,
            videoTrueView = 0,
            avgNShow = 0.0,
            avgNShowComplete = 0.0,
        )
    }

    fun ratio(num: Double, divisor: Double): Double? = when (divisor) {
        0.0 -> null
        else -> (num / divisor).takeIf { it.isFinite() }
    }

    fun ratioFromMicros(numMicros: Long, divisor: Long): BigDecimal? = when (divisor) {
        0L -> null
        else -> fromMicros(numMicros / divisor)
    }
}
