package ru.yandex.direct.web.entity.uac.converter

import java.math.BigDecimal
import java.time.LocalDate
import java.util.EnumSet
import ru.yandex.direct.core.entity.campaign.model.CampaignType.MOBILE_CONTENT
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.MobileContentCampaign
import ru.yandex.direct.core.entity.campaign.model.SmsFlag
import ru.yandex.direct.core.entity.campaign.model.StrategyData
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CLICK
import ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA
import ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPI
import ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_CRR
import ru.yandex.direct.core.entity.campaign.model.StrategyName.toSource
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppAlternativeStore
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDeviceTypeTargeting
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppNetworkTargeting
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker
import ru.yandex.direct.core.entity.uac.converter.UacCampaignConverter.toCampaignsPlatform
import ru.yandex.direct.core.entity.uac.model.AltAppStore
import ru.yandex.direct.core.entity.uac.model.AltAppStore.Companion.toCoreType
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.TargetType
import ru.yandex.direct.core.entity.uac.model.TrackingUrl
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.libs.timetarget.TimeTargetUtils
import ru.yandex.direct.utils.mapToSet
import ru.yandex.direct.web.entity.uac.model.UacModifyCampaignDataContainer

object UacMobileCampaignConverter {

    private fun toStrategyData(
        strategyName: StrategyName,
        dataContainer: UacModifyCampaignDataContainer
    ): StrategyData? {
        val strategyData = StrategyData()
            .withName(toSource(strategyName)!!.literal)
            .withSum(dataContainer.weekLimit)
            .withVersion(1L)

        if (dataContainer.strategy != null) {
            return when (strategyName) {
                AUTOBUDGET_AVG_CPA -> strategyData
                    .withPayForConversion(dataContainer.strategy.uacStrategyData.payForConversion)
                    .withAvgCpa(dataContainer.strategy.uacStrategyData.avgCpa)
                    .withGoalId(dataContainer.targetId!!.goalId)
                AUTOBUDGET_AVG_CPI -> strategyData
                    .withPayForConversion(dataContainer.strategy.uacStrategyData.payForConversion)
                    .withAvgCpi(dataContainer.strategy.uacStrategyData.avgCpi)
                    .withGoalId(dataContainer.targetId!!.goalId)
                AUTOBUDGET_AVG_CLICK -> strategyData
                    .withAvgBid(dataContainer.strategy.uacStrategyData.avgBid)
                AUTOBUDGET_CRR -> strategyData
                    .withPayForConversion(dataContainer.strategy.uacStrategyData.payForConversion)
                    .withCrr(dataContainer.strategy.uacStrategyData.crr)
                    .withGoalId(dataContainer.targetId!!.goalId)
                else -> throw IllegalStateException("Illegal Strategy name $strategyName")
            }
        }
        return deprecatedToStrategyData(strategyName, dataContainer, strategyData)
    }

    // Old method for interoperability until migration
    private fun deprecatedToStrategyData(
        strategyName: StrategyName,
        dataContainer: UacModifyCampaignDataContainer,
        strategyData: StrategyData
    ): StrategyData = when (strategyName) {
        AUTOBUDGET_AVG_CPA -> strategyData
            .withPayForConversion(true)
            .withAvgCpa(dataContainer.cpa)
            .withGoalId(dataContainer.targetId!!.goalId)
        AUTOBUDGET_AVG_CPI -> strategyData
            .withPayForConversion(dataContainer.targetId == TargetType.INSTALL)
            .withAvgCpi(dataContainer.cpa)
            .withGoalId(dataContainer.targetId!!.goalId)
        AUTOBUDGET_AVG_CLICK -> strategyData
            .withAvgBid(dataContainer.cpa)
        else -> throw IllegalStateException("Illegal Strategy name $strategyName")
    }


    private fun toStrategy(
        dataContainer: UacModifyCampaignDataContainer,
        enabledFeaturesForClient: MutableSet<String>
    ): DbStrategy {
        val strategyName = if (dataContainer.strategy != null) {
            dataContainer.strategy.uacStrategyName.toDbStrategyName()
        } else if (dataContainer.targetId == TargetType.CPC || dataContainer.skadNetworkEnabled == true) {
            AUTOBUDGET_AVG_CLICK
        } else if (enabledFeaturesForClient.contains(FeatureName.UAC_FIX_CPA_STRATEGY_ENABLED.getName())) {
            AUTOBUDGET_AVG_CPA
        } else {
            AUTOBUDGET_AVG_CPI
        }

        return DbStrategy().apply {
            withPlatform(toCampaignsPlatform(dataContainer.strategyPlatform))
            withStrategyName(strategyName)
            withAutobudget(CampaignsAutobudget.YES)
            withStrategyData(toStrategyData(strategyName, dataContainer))
        }
    }

    fun toMobileApp(appInfo: AppInfo, trackingUrl: TrackingUrl?, impressionUrl: TrackingUrl?): MobileApp {
        val mobileApp = MobileApp()
            .withName(appInfo.title)
            .withStoreHref(appInfo.url)
            .withDisplayedAttributes(setOf())

        return if (trackingUrl == null)
            mobileApp.withTrackers(listOf())
        else
            mobileApp.withTrackers(
                listOf(
                    MobileAppTracker()
                        .withTrackingSystem(trackingUrl.system.directMobileTrackingSystem)
                        .withUrl(trackingUrl.getUrl())
                        .withImpressionUrl(impressionUrl?.getUrl() ?: "")
                        .withUserParams(listOf())
                )
            )
    }

    fun toMobileContentCampaign(
        request: UacModifyCampaignDataContainer,
        user: User,
        enabledFeaturesForClient: MutableSet<String>,
        disabledDomains: List<String>?,
    ): MobileContentCampaign {
        val campaign = MobileContentCampaign()
            .withType(MOBILE_CONTENT)
            .withName(request.displayName)
            .withGeo(setOf())
            .withEnableCpcHold(CampaignConstants.DEFAULT_ENABLE_CPC_HOLD)
            .withDeviceTypeTargeting(EnumSet.allOf(MobileAppDeviceTypeTargeting::class.java))
            .withDayBudgetShowMode(DayBudgetShowMode.DEFAULT_)
            .withHasExtendedGeoTargeting(false)
            .withSmsFlags(EnumSet.noneOf(SmsFlag::class.java))
            .withNetworkTargeting(EnumSet.allOf(MobileAppNetworkTargeting::class.java))
            .withSmsTime(CampaignConstants.DEFAULT_SMS_TIME_INTERVAL)
            .withStartDate(LocalDate.now())
            .withTimeZoneId(TimeTargetUtils.DEFAULT_TIMEZONE)
            .withDayBudget(BigDecimal.valueOf(0L))
            .withIsSkadNetworkEnabled(request.skadNetworkEnabled)
            .withIsAllowedOnAdultContent(request.adultContentEnabled)
            .withStrategy(toStrategy(request, enabledFeaturesForClient))
            .withSource(request.source)
            .withEmail(user.email)
            .withMobileAppId(request.mobileAppId)
            .withWidgetPartnerId(request.widgetPartnerId)
            .withDisabledDomains(disabledDomains)
            .withAlternativeAppStores(toMobileAppAlternativeStores(request.altAppStores))
            .withUseCurrentRegion(request.useCurrentRegion)
            .withUseRegularRegion(request.useRegularRegion)
        return campaign;
    }

    private fun toMobileAppAlternativeStores(altAppStores: Set<AltAppStore>?): EnumSet<MobileAppAlternativeStore>? {
        if (altAppStores.isNullOrEmpty()) {
            return null
        }
        return EnumSet.copyOf(altAppStores.mapToSet { it.toCoreType() })
    }

}
