package ru.yandex.direct.web.entity.uac.converter

import ru.yandex.direct.core.entity.retargeting.Constants
import ru.yandex.direct.core.entity.retargeting.Constants.MAX_GOAL_TIME
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.retargeting.model.RuleType
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRule
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoal
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoalType
import ru.yandex.direct.core.entity.uac.model.UacSegmentInfo
import ru.yandex.direct.web.core.model.retargeting.AbstractGoalWeb
import ru.yandex.direct.web.core.model.retargeting.Condition
import ru.yandex.direct.web.core.model.retargeting.MetrikaGoalWeb
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionRuleType
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionType
import ru.yandex.direct.web.core.model.retargeting.RetargetingConditionWeb
import ru.yandex.direct.web.entity.uac.model.RetargetingConditionRequest
import ru.yandex.direct.web.entity.uac.model.UacMobileGoal

object UacRetargetingConverter {
    fun toUacMobileGoal(goal: Goal): UacMobileGoal {
        return UacMobileGoal(goalId = goal.id, goalName = goal.name)
    }

    fun toUacRetargetingCondition(
        retargetingCondition: RetargetingCondition,
        goalNameById: Map<Long, String>,
        lalSegmentIdToParentId: Map<Long, Long>,
        metrikaSegmentIdToGoal: Map<Long, Goal> = mapOf(),
    ): UacRetargetingCondition {
        return UacRetargetingCondition(
            name = retargetingCondition.name,
            conditionRules = retargetingCondition.rules.map {
                UacRetargetingConditionRule(
                    type = toUacRuleType(it.type),
                    goals = it.goals.map { goal ->
                        toUacGoal(
                            goal,
                            goalNameById,
                            lalSegmentIdToParentId,
                            metrikaSegmentIdToGoal
                        )
                    }
                )
            },
            id = retargetingCondition.id,
        )
    }

    private fun toUacRuleType(type: RuleType): UacRetargetingConditionRule.RuleType {
        return when (type) {
            RuleType.OR -> UacRetargetingConditionRule.RuleType.OR
            RuleType.ALL -> UacRetargetingConditionRule.RuleType.ALL
            RuleType.NOT -> UacRetargetingConditionRule.RuleType.NOT
        }
    }

    private fun toUacGoal(
        goal: Goal,
        goalNameById: Map<Long, String>,
        lalSegmentIdToParentId: Map<Long, Long>,
        metrikaSegmentIdToGoal: Map<Long, Goal>,
    ): UacRetargetingConditionRuleGoal {
        return if (
            goal.type == GoalType.LAL_SEGMENT &&
            Goal.computeType(lalSegmentIdToParentId[goal.id]) in listOf(GoalType.MOBILE, GoalType.AUDIENCE)
        ) {
            val type = if (Goal.computeType(lalSegmentIdToParentId[goal.id]) == GoalType.MOBILE) {
                UacRetargetingConditionRuleGoalType.LAL
            } else {
                UacRetargetingConditionRuleGoalType.LAL_AUDIENCE
            }
            UacRetargetingConditionRuleGoal(
                id = lalSegmentIdToParentId[goal.id],
                time = Constants.MAX_GOAL_TIME,
                name = goalNameById[lalSegmentIdToParentId[goal.id]],
                type = type,
            )
        } else if (
            goal.type == GoalType.LAL_SEGMENT
            && Goal.computeType(lalSegmentIdToParentId[goal.id]) == GoalType.SEGMENT
        ) {
            UacRetargetingConditionRuleGoal(
                id = lalSegmentIdToParentId[goal.id]!!,
                time = Constants.MAX_GOAL_TIME,
                name = goalNameById[lalSegmentIdToParentId[goal.id]],
                type = UacRetargetingConditionRuleGoalType.LAL_SEGMENT,
                segmentInfo = UacSegmentInfo(
                    counterId = metrikaSegmentIdToGoal[lalSegmentIdToParentId[goal.id]]?.counterId,
                    name = metrikaSegmentIdToGoal[lalSegmentIdToParentId[goal.id]]?.name,
                    domain = metrikaSegmentIdToGoal[lalSegmentIdToParentId[goal.id]]?.domain,
                )
            )
        } else if (goal.type == GoalType.SEGMENT) {
            UacRetargetingConditionRuleGoal(
                id = goal.id,
                time = goal.time,
                name = goalNameById[goal.id],
                type = UacRetargetingConditionRuleGoalType.SEGMENT,
                segmentInfo = UacSegmentInfo(
                    counterId = metrikaSegmentIdToGoal[goal.id]?.counterId,
                    name = metrikaSegmentIdToGoal[goal.id]?.name,
                    domain = metrikaSegmentIdToGoal[goal.id]?.domain,
                )
            )
        } else {
            val type = if (Goal.computeType(goal.id) == GoalType.MOBILE) {
                UacRetargetingConditionRuleGoalType.MOBILE
            } else {
                UacRetargetingConditionRuleGoalType.AUDIENCE
            }
            UacRetargetingConditionRuleGoal(
                id = goal.id,
                time = goal.time,
                name = goalNameById[goal.id],
                type = type,
            )
        }
    }

    fun toRetargetingConditionWeb(
        request: RetargetingConditionRequest,
        clientId: Long,
        retargetingConditionId: Long,
        parentIdToLalSegmentId: Map<Long, Long>,
    ) = RetargetingConditionWeb()
        .withRetargetingConditionId(retargetingConditionId)
        .withConditionName(request.name)
        .withClientId(clientId)
        .withType(RetargetingConditionType.metrika_goals)
        .withConditions(toConditions(request.conditionRules, parentIdToLalSegmentId))

    private fun toConditions(
        rules: List<UacRetargetingConditionRule>,
        parentIdToLalSegmentId: Map<Long, Long>,
    ): List<Condition> {
        return rules.map {
            Condition()
                .withConditionType(toRetargetingConditionRuleType(it.type))
                .withConditionGoalWebs(toGoalWebs(it.goals, parentIdToLalSegmentId))
        }
    }

    private fun toRetargetingConditionRuleType(
        type: UacRetargetingConditionRule.RuleType
    ): RetargetingConditionRuleType {
        return when (type) {
            UacRetargetingConditionRule.RuleType.OR -> RetargetingConditionRuleType.or
            UacRetargetingConditionRule.RuleType.ALL -> RetargetingConditionRuleType.all
            UacRetargetingConditionRule.RuleType.NOT -> RetargetingConditionRuleType.not
        }
    }

    private fun toGoalWebs(
        goals: List<UacRetargetingConditionRuleGoal>,
        parentIdToLalSegmentId: Map<Long, Long>,
    ): List<AbstractGoalWeb> {
        return goals.map { goal ->
            toGoalWeb(goal, parentIdToLalSegmentId)
        }
    }

    private fun toGoalWeb(
        goal: UacRetargetingConditionRuleGoal,
        parentIdToLalSegmentId: Map<Long, Long>,
    ): AbstractGoalWeb {
        return when (goal.type) {
            UacRetargetingConditionRuleGoalType.MOBILE -> toWebGoal(goal, goal.time!!)
            UacRetargetingConditionRuleGoalType.AUDIENCE -> toWebGoal(goal, MAX_GOAL_TIME)
            UacRetargetingConditionRuleGoalType.LAL -> toLalWebGoal(goal, parentIdToLalSegmentId)
            UacRetargetingConditionRuleGoalType.LAL_AUDIENCE -> toLalWebGoal(goal, parentIdToLalSegmentId)
            UacRetargetingConditionRuleGoalType.SEGMENT -> toSegmentWebGoal(goal)
            UacRetargetingConditionRuleGoalType.LAL_SEGMENT -> toSegmentLalWebGoal(goal, parentIdToLalSegmentId)
            else -> toWebGoal(goal, goal.time)
        }
    }

    private fun toWebGoal(uacGoal: UacRetargetingConditionRuleGoal, time: Int?) = MetrikaGoalWeb()
        .withId(uacGoal.id)
        .withTime(time)
        .withName(uacGoal.name)

    private fun toLalWebGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
        parentIdToLalSegmentId: Map<Long, Long>
    ) = MetrikaGoalWeb()
        .withId(parentIdToLalSegmentId[uacGoal.id])
        .withTime(MAX_GOAL_TIME)

    private fun toSegmentWebGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
    ) = MetrikaGoalWeb()
        .withId(uacGoal.id)
        .withTime(MAX_GOAL_TIME)
        .withCounterId(uacGoal.segmentInfo?.counterId)
        .withCounterName(uacGoal.segmentInfo?.name)
        .withDomain(uacGoal.segmentInfo?.domain)

    private fun toSegmentLalWebGoal(
        uacGoal: UacRetargetingConditionRuleGoal,
        parentIdToLalSegmentId: Map<Long, Long>,
    ) = MetrikaGoalWeb()
        .withId(parentIdToLalSegmentId[uacGoal.id])
        .withTime(MAX_GOAL_TIME)
        .withCounterId(uacGoal.segmentInfo?.counterId)
        .withCounterName(uacGoal.segmentInfo?.name)
        .withDomain(uacGoal.segmentInfo?.domain)
}
