package ru.yandex.direct.web.entity.uac.converter

import java.math.BigDecimal
import java.math.MathContext
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.core.entity.uac.UacCommonUtils.CREATIVE_ID_KEY
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toFeedFilterValue
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.DeviceType
import ru.yandex.direct.core.entity.uac.model.LimitPeriodType
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Sitelink
import ru.yandex.direct.core.entity.uac.model.Socdem
import ru.yandex.direct.core.entity.uac.model.UacFeedFilter
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterCondition
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterOperator
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.UacGeoService.getGeoForUacGroups
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.currency.CurrencyCode
import ru.yandex.direct.grid.model.campaign.facelift.GdBudgetDisplayFormat
import ru.yandex.direct.grid.processing.model.banner.mutation.GdAddSitelink
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdAddUcCampaignInput
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdMeaningfulGoalRequest
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcCampaignMutationInput
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcFeedFilter
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcFeedFilterCondition
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUcFeedFilterOperator
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdUpdateUcCampaignInput
import ru.yandex.direct.grid.processing.model.forecast.GdDeviceType
import ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemAgePoint
import ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemGender
import ru.yandex.direct.grid.processing.model.touchsocdem.GdTouchSocdemInput
import ru.yandex.direct.utils.FunctionalUtils.mapList
import ru.yandex.direct.web.entity.uac.converter.UacTimeTargetConverter.toGdTimeTarget
import ru.yandex.direct.web.entity.uac.model.CreateCampaignInternalRequest
import ru.yandex.direct.web.entity.uac.model.PatchCampaignInternalRequest
import ru.yandex.direct.web.entity.uac.model.UacModifyCampaignDataContainer
import ru.yandex.direct.web.entity.uac.model.UacStrategyDataContainer

object UacTextCampaignConverter {

    fun toGdAddUcCampaignInput(
        createDataContainer: UacModifyCampaignDataContainer,
        contents: Collection<Content>,
        currencyCode: CurrencyCode,
    ): GdAddUcCampaignInput {
        val strategyDataContainer = toStrategyDataContainer(createDataContainer)
        val imageContent = contents.find { it.type == MediaType.IMAGE }
        val videoContent = contents.find { it.type == MediaType.VIDEO }
        return GdAddUcCampaignInput()
            .withName(createDataContainer.displayName)
            .withHref(createDataContainer.href)
            .withPermalinkId(createDataContainer.permalinkId)
            .withPhoneId(createDataContainer.phoneId)
            .withAdTitle(createDataContainer.titles!![0])
            .withAdText(createDataContainer.texts!![0])
            .withSitelinks(createDataContainer.sitelinks?.map { toGdSitelink(it) })
            .withGeo(getGeoForUacGroups(createDataContainer.regions, createDataContainer.minusRegions))
            .withHyperGeoId(createDataContainer.hyperGeoId)
            .withKeywords(createDataContainer.keywords)
            .withSocdem(toGdSocdem(createDataContainer.socdem))
            .withDeviceTypes(createDataContainer.deviceTypes?.mapTo(HashSet()) { toGdDeviceType(it) })
            .withGoalId(getGoalId(strategyDataContainer))
            .withMeaningfulGoals(getMeaningfulGoals(strategyDataContainer, currencyCode))
            .withAvgCpa(createDataContainer.cpa)
            .withCrr(getCrr(strategyDataContainer))
            .withBudget(createDataContainer.weekLimit)
            .withBudgetDisplayFormat(createDataContainer.limitPeriod.toBudgetDisplayFormat())
            .withMetrikaCounters(createDataContainer.counters)
            .withIsDraft(true)
            .withImageHash(imageContent?.directImageHash)
            .withCreativeId(getCreativeId(videoContent))
            .withCalltrackingSettingsId(createDataContainer.calltrackingSettingsId)
            .withTimeTarget(createDataContainer.timeTarget?.toGdTimeTarget())
            .withIsEcom(createDataContainer.isEcom != null && createDataContainer.isEcom)
            .withFeedId(createDataContainer.feedId)
            .withFeedFilters(toGdFeedFilters(createDataContainer.feedFilters))
            .withTrackingParams(createDataContainer.trackingParams)
            .withSource(createDataContainer.source)
            .withWidgetPartnerId(createDataContainer.widgetPartnerId)
            .withIsRecommendationsManagementEnabled(createDataContainer.isRecommendationsManagementEnabled)
            .withIsPriceRecommendationsManagementEnabled(createDataContainer.isPriceRecommendationsManagementEnabled)
    }

    fun toGdUpdateUcCampaignInput(
        updatedUacYdbCampaign: UacYdbCampaign,
        campaign: TextCampaign,
        request: PatchCampaignInternalRequest,
        contents: Collection<Content>,
        currencyCode: CurrencyCode,
    ): GdUpdateUcCampaignInput {
        val strategyDataContainer = toStrategyDataContainer(request)
        val adTitle = request.titles?.get(0)
        val adText = request.texts?.get(0)
        val imageContent = contents.find { it.type == MediaType.IMAGE }
        val videoContent = contents.find { it.type == MediaType.VIDEO }

        return GdUpdateUcCampaignInput()
            .withId(campaign.id)
            .withName(updatedUacYdbCampaign.name)
            .withHref(updatedUacYdbCampaign.storeUrl)
            .withPermalinkId(updatedUacYdbCampaign.permalinkId)
            .withPhoneId(updatedUacYdbCampaign.phoneId)
            .withGeo(getGeoForUacGroups(updatedUacYdbCampaign.regions, updatedUacYdbCampaign.minusRegions))
            .withHyperGeoId(updatedUacYdbCampaign.hyperGeoId)
            .withKeywords(updatedUacYdbCampaign.keywords)
            .withSocdem(toGdSocdem(updatedUacYdbCampaign.socdem))
            .withDeviceTypes(updatedUacYdbCampaign.deviceTypes?.mapTo(HashSet()) { toGdDeviceType(it) })
            .withGoalId(getGoalId(strategyDataContainer))
            .withMeaningfulGoals(getMeaningfulGoals(strategyDataContainer, currencyCode))
            .withAvgCpa(updatedUacYdbCampaign.cpa)
            .withCrr(getCrr(strategyDataContainer))
            .withBudget(updatedUacYdbCampaign.weekLimit)
            .withBudgetDisplayFormat(updatedUacYdbCampaign.options?.limitPeriod.toBudgetDisplayFormat())
            .withMetrikaCounters(updatedUacYdbCampaign.counters)
            .withKeywords(updatedUacYdbCampaign.keywords)
            .withAdTitle(adTitle)
            .withAdText(adText)
            .withImageHash(imageContent?.directImageHash)
            .withCreativeId(getCreativeId(videoContent))
            .withCalltrackingSettingsId(updatedUacYdbCampaign.calltrackingSettingsId)
            .withSitelinks(request.sitelinks?.map { toGdSitelink(it) })
            .withTimeTarget(updatedUacYdbCampaign.timeTarget?.toGdTimeTarget())
            .withIsEcom(updatedUacYdbCampaign.isEcom)
            .withFeedId(updatedUacYdbCampaign.feedId)
            .withFeedFilters(toGdFeedFilters(updatedUacYdbCampaign.feedFilters))
            .withTrackingParams(updatedUacYdbCampaign.trackingParams)
            .withIsRecommendationsManagementEnabled(updatedUacYdbCampaign.recommendationsManagementEnabled)
            .withIsPriceRecommendationsManagementEnabled(updatedUacYdbCampaign.priceRecommendationsManagementEnabled)
    }

    fun toStrategyDataContainer(request: CreateCampaignInternalRequest) = request.run {
        UacStrategyDataContainer(crr, goals, weekLimit, cpa)
    }

    fun toStrategyDataContainer(request: PatchCampaignInternalRequest) = request.run {
        UacStrategyDataContainer(crr, goals, weekLimit, cpa)
    }

    fun toStrategyDataContainer(request: UacModifyCampaignDataContainer) = request.run {
        UacStrategyDataContainer(crr, goals, weekLimit, cpa)
    }

    // Выбираем оплату за конверсии, если задана цена конверсии на общем уровне.
    // При этом цель должна быть строго одна, это должно проверяться валидацией.
    fun isCpaStrategy(strategyDataContainer: UacStrategyDataContainer) = strategyDataContainer.run {
        cpa != null
    }

    // Выбираем ДРР, когда либо задана ДРР (при этом у части целей может быть задана цена cpa,
    // тогда мы зададим для них такую фиксированную ценность, чтобы доля от нее соответствовала переданной цене),
    // либо же у целей заданы цены конверсий cpa (тогда сами выберем ДРР и подгоним ценности).
    // Кейс, когда не задана ДРР и при этом у некоторых целей не заданы cpa, должен отсекаться валидацией.
    // Должна быть задана хотя бы одна цель.
    fun isCrrStrategy(strategyDataContainer: UacStrategyDataContainer) = strategyDataContainer.run {
        crr != null || goals?.any { it.cpa != null } ?: false
    }

    fun getCrr(strategyDataContainer: UacStrategyDataContainer) = strategyDataContainer.run {
        if (isCrrStrategy(this)) crr ?: 100 else null
    }

    fun getGoalId(strategyDataContainer: UacStrategyDataContainer) = strategyDataContainer.goals.run {
        when {
            isNullOrEmpty() -> null
            size == 1 -> first().goalId
            else -> CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID
        }
    }

    fun getMeaningfulGoals(
        strategyDataContainer: UacStrategyDataContainer,
        currencyCode: CurrencyCode
    ): List<GdMeaningfulGoalRequest>? = strategyDataContainer.goals.run {
        when {
            isNullOrEmpty() -> null
            size == 1 -> null
            else -> {
                val fallbackConversionValue = Currencies.getCurrency(currencyCode).ucDefaultConversionValue

                val crr = getCrr(strategyDataContainer)?.let { BigDecimal.valueOf(it, 2) }
                return if (crr != null) {
                    map {
                        GdMeaningfulGoalRequest().apply {
                            goalId = it.goalId
                            conversionValue = it.cpa?.divide(crr, MathContext.DECIMAL128) ?: fallbackConversionValue
                            isMetrikaSourceOfValue = it.cpa == null
                        }
                    }
                } else {
                    map {
                        GdMeaningfulGoalRequest().apply {
                            goalId = it.goalId
                            conversionValue = it.conversionValue ?: fallbackConversionValue
                        }
                    }
                }
            }
        }
    }

    fun getCreativeId(videoContent: Content?) =
        (videoContent?.meta?.get(CREATIVE_ID_KEY) as? Number)?.toLong()

    fun getGoalIds(input: GdUcCampaignMutationInput): List<Long?>? {
        if (input.goalId == null) {
            return null
        }
        return if (input.goalId == CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID) {
            input.meaningfulGoals.map { it.goalId }
        } else {
            listOf(input.goalId)
        }
    }

    private fun toGdSitelink(sitelink: Sitelink?): GdAddSitelink? {
        if (sitelink == null) {
            return null
        }
        return GdAddSitelink()
            .withTitle(sitelink.title)
            .withHref(sitelink.href)
            .withDescription(sitelink.description)
    }

    private fun toGdFeedFilters(feedFilters: List<UacFeedFilter>?): List<GdUcFeedFilter>? {
        if (feedFilters == null) {
            return null
        }
        return mapList(feedFilters, UacTextCampaignConverter::toGdFeedFilter)
    }

    private fun toGdFeedFilter(feedFilter: UacFeedFilter): GdUcFeedFilter {
        return GdUcFeedFilter()
                .withConditions(mapList(feedFilter.conditions, UacTextCampaignConverter::toGdFeedFilterCondition))
    }

    private fun toGdFeedFilterCondition(condition: UacFeedFilterCondition): GdUcFeedFilterCondition {
        return GdUcFeedFilterCondition()
                .withField(condition.field)
                .withOperator(toGdFeedFilterOperator(condition.operator))
                .withValue(condition.values?.toFeedFilterValue() ?: condition.value)
    }

    private fun toGdFeedFilterOperator(operator: UacFeedFilterOperator): GdUcFeedFilterOperator = when(operator) {
        UacFeedFilterOperator.GREATER -> GdUcFeedFilterOperator.GREATER
        UacFeedFilterOperator.LESS -> GdUcFeedFilterOperator.LESS
        UacFeedFilterOperator.EQUALS -> GdUcFeedFilterOperator.EQUALS
        UacFeedFilterOperator.RANGE -> GdUcFeedFilterOperator.RANGE
        UacFeedFilterOperator.CONTAINS -> GdUcFeedFilterOperator.CONTAINS
        UacFeedFilterOperator.NOT_CONTAINS -> GdUcFeedFilterOperator.NOT_CONTAINS
        UacFeedFilterOperator.EXISTS -> GdUcFeedFilterOperator.EXISTS
        UacFeedFilterOperator.UNKNOWN -> GdUcFeedFilterOperator.UNKNOWN
    }

    private fun toGdSocdem(socdem: Socdem?): GdTouchSocdemInput? {
        if (socdem == null) {
            return null
        }
        return GdTouchSocdemInput()
            .withGenders(socdem.genders.map { GdTouchSocdemGender.valueOf(it.name.uppercase()) })
            .withAgeLower(GdTouchSocdemAgePoint.valueOf(socdem.ageLower.name.uppercase()))
            .withAgeUpper(GdTouchSocdemAgePoint.valueOf(socdem.ageUpper.name.uppercase()))
    }

    private fun toGdDeviceType(deviceType: DeviceType) = GdDeviceType.valueOf(deviceType.name.uppercase())

    private fun LimitPeriodType?.toBudgetDisplayFormat(): GdBudgetDisplayFormat {
        return when (this) {
            null, LimitPeriodType.WEEK -> GdBudgetDisplayFormat.WEEKLY
            LimitPeriodType.MONTH -> GdBudgetDisplayFormat.MONTHLY
        }
    }

}
