package ru.yandex.direct.web.entity.uac.converter.proto

import org.assertj.core.api.recursive.comparison.ComparisonDifference
import org.assertj.core.api.recursive.comparison.RecursiveComparisonConfiguration
import org.assertj.core.api.recursive.comparison.RecursiveComparisonDifferenceCalculator
import org.slf4j.LoggerFactory
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import ru.yandex.direct.web.entity.uac.model.UacProtoResponse

typealias ResponseFunc = () -> ResponseEntity<Any>

/**
 * Класс для поиска недостающих данных в proto-формате ручек UAC.
 * Формирует ответ в обычном формате и в proto-формате, пишет в лог различия.
 *
 * @param responseMapper    маппер ответа из proto-формата в обычный формат
 * @param alias             дописывается в лог, чтобы можно было понять к какой ручке относится запись
 * @param percentGetter     лямбда для получения числа в интервале [0..100] - процент, на который прогонять обе ветки
 * @param recursiveComparisonConfiguration конфиг для RecursiveComparisonDifferenceCalculator
 */
class ProtoResponseDiffer<From : UacProtoResponse<*>, To>(
    private val responseMapper: UacProtoResponseMapper<From, To>,
    private val alias: String,
    private val percentGetter: () -> Int,
    private val recursiveComparisonConfiguration: RecursiveComparisonConfiguration,
    private val fixOldResponse: (To) -> To = { it },
) {
    companion object {
        private val logger = LoggerFactory.getLogger(ProtoResponseDiffer::class.java)
    }

    data class Result(
        val response: ResponseEntity<Any>,
        val diff: List<ComparisonDifference>? = null,
    )

    fun process(
        oldResponseCreator: ResponseFunc,
        protoResponseCreator: ResponseFunc,
    ): Result {
        val oldResponse = oldResponseCreator()

        if (oldResponse.statusCode != HttpStatus.OK) {
            return Result(oldResponse)
        }

        if ((0..99).random() >= percentGetter()) {
            return Result(oldResponse)
        }

        val protoResponse = try {
            protoResponseCreator()
        } catch (e: Exception) {
            logger.error("Could not create proto response ({}): {}", alias, e.message)
            return Result(oldResponse)
        }

        if (protoResponse.statusCode != oldResponse.statusCode) {
            logger.warn("Different response statuses ({}): {} (proto) vs {} (old)",
                alias, protoResponse.statusCode, oldResponse.statusCode)
            return Result(oldResponse)
        }

        val protoResponseBody: From?
        val oldResponseBody: To?
        try {
            protoResponseBody = protoResponse.body as From
            oldResponseBody = oldResponse.body as To
        } catch (e: ClassCastException) {
            logger.error("Unexpected response body type ({}): {}", alias, e.message)
            return Result(oldResponse)
        }

        val oldMappedResponseBody = try {
            responseMapper.convert(protoResponseBody)
        } catch (e: Exception) {
            logger.error("Could not map proto response to old format ({}): {}", alias, e.message)
            return Result(oldResponse)
        }

        val diff = RecursiveComparisonDifferenceCalculator().determineDifferences(
            oldMappedResponseBody,
            fixOldResponse(oldResponseBody),
            recursiveComparisonConfiguration,
        )

        val significantDiff = dropDefaultValuesDiff(diff)
        if (significantDiff.isNotEmpty()) {
            logger.warn("Found diff between response formats ({}):{}{}",
                alias, '\n', significantDiff.joinToString(",\n"))
        } else {
            logger.info("Responses are equivalent ({})", alias)
        }
        return Result(oldResponse, significantDiff)
    }

    private fun dropDefaultValuesDiff(diff: List<ComparisonDifference>): List<ComparisonDifference> {
        return diff.filter { !isEmpty(it.actual) || !isEmpty(it.expected) }
    }

    private fun isEmpty(value: Any?) = when (value) {
        null -> true
        is Collection<*> -> value.isEmpty()
        is Map<*, *> -> value.isEmpty()
        is Number -> value.toDouble() == 0.0
        is Boolean -> value == false
        is String -> value.isEmpty()
        else -> false
    }
}
