package ru.yandex.direct.web.entity.uac.converter.proto

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.node.ArrayNode
import com.fasterxml.jackson.databind.node.ObjectNode
import com.fasterxml.jackson.module.kotlin.readValue
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toAdvType
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toContentFlags
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toCpmAsset
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toDeviceTypes
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toInventoryTypes
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toLimitPeriodType
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toRelevanceMatch
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toRetargetingCondition
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toShowsFrequencyLimit
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toSocdem
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toTargetStatus
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toTargetType
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toTimeTarget
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacBrandSafety
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacDisabledPlaces
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacFeedFilter
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacGoals
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacSearchLift
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacStrategy
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacStrategyPlatform
import ru.yandex.direct.core.entity.uac.model.AltAppStore.Companion.fromCoreType
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.LimitPeriodType
import ru.yandex.direct.core.entity.uac.model.Sitelink
import ru.yandex.direct.core.entity.uac.model.UacAdjustment
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCampaignMeasurer
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCampaignMeasurerSystem
import ru.yandex.direct.core.grut.api.utils.uint64ToColorString
import ru.yandex.direct.currency.Money.MONEY_CENT_SCALE
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.web.entity.uac.converter.proto.enummappers.UacEnumMappers
import ru.yandex.direct.web.entity.uac.model.UacCampaign
import ru.yandex.direct.web.entity.uac.model.UacCampaignAccess
import ru.yandex.direct.web.entity.uac.model.UacCampaignAgencyInfo
import ru.yandex.direct.web.entity.uac.model.UacCampaignManagerInfo
import ru.yandex.direct.web.entity.uac.model.UacCampaignProtoResponse
import ru.yandex.direct.web.entity.uac.model.UacCampaignResponse
import ru.yandex.direct.web.entity.uac.service.BaseUacCampaignWebService
import java.math.RoundingMode

class UacGetCampaignProtoResponseMapper(objectMapper: ObjectMapper) :
    UacProtoResponseMapper<UacCampaignProtoResponse, UacCampaignResponse>(objectMapper)
{
    private val logger = LoggerFactory.getLogger(UacGetCampaignProtoResponseMapper::class.java)

    override fun logger(): Logger = logger

    override fun convert(from: UacCampaignProtoResponse): UacCampaignResponse {
        val convertedResponse = UacCampaignResponse(from.reqId)
        val result = from.result ?: return convertedResponse

        val jsonTree = objectMapper.readValue<JsonNode>(
            objectMapper.writeValueAsString(from)
        )
        val campaignMeta = result.campaign.meta
        val campaignSpec = result.campaign.spec
        val brief = campaignSpec.campaignBrief

        val accessKt = (jsonTree.get("result").get("access") as ObjectNode).let { node ->
            addEmptyArrayIfMissing(node, "actions", objectMapper)
            addEmptyArrayIfMissing(node, "pseudo_actions", objectMapper)
            convertEnumArray(node.get("actions"))
            convertEnumArray(node.get("pseudo_actions"))
            replaceEnumValueNode(node, "serviced_state")

            treeToValueSafe(node, UacCampaignAccess::class.java)
                ?: throw IllegalArgumentException("Bad input for UacCampaignAccess")
        }

        val campaignAgencyInfoKt = (jsonTree.get("result").get("agency_info") as? ObjectNode)?.let {
            treeToValueSafe(it, UacCampaignAgencyInfo::class.java)
        }

        val managerInfoKt = (jsonTree.get("result").get("manager_info") as? ObjectNode)?.let {
            treeToValueSafe(it, UacCampaignManagerInfo::class.java)
        }

        val appInfo = (jsonTree.get("result").get("app_info") as? ObjectNode)?.let { node ->
            node.get("app_info")?.get("meta")?.let {
                node.put("id", it.get("id").textValue())
            }
            node.get("app_info")?.get("spec")?.let { spec ->
                for (key in listOf("app_id", "bundle_id", "language", "region", "platform")) {
                    spec.get(key)?.let { node.set(key, it) }
                }
            }
            node.remove("app_info")
            treeToValueSafe(node, AppInfo::class.java)
        }

        val hyperGeo = (jsonTree.get("result").get("hyper_geo") as? ObjectNode)?.let { node ->
            (node.get("hyper_geo_segments") as? ArrayNode)?.forEach { segmentNode ->
                replaceEnumValueNode(segmentNode.get("hyper_geo_segment_details"), "geo_segment_type")
            }
            treeToValueSafe(node, HyperGeo::class.java)
        }

        val sitelinks = (jsonTree.get("result").get("sitelinks") as? ArrayNode)?.mapNotNull {
            treeToValueSafe(it, Sitelink::class.java)
        }

        val brandSurveyStatus = (jsonTree.get("result").get("brand_survey_status") as? ObjectNode)?.let { node ->
            replaceEnumValueNode(node, "survey_status_daily", lowercase = false)
            convertEnumArray(node.get("brand_survey_stop_reasons_daily"), lowercase = false)
            treeToValueSafe(node, BrandSurveyStatus::class.java)
        }

        val adjustments = (jsonTree.get("result").get("adjustments") as? ArrayNode)?.mapNotNull {
            val node = it as ObjectNode
            val nested = node.get("adjustment") as ObjectNode
            replaceEnumValueNode(nested, "age")
            replaceEnumValueNode(nested, "gender")
            for ((k, v) in nested.fields()) {
                node.set(k, v)
            }
            node.remove("adjustment")
            treeToValueSafe(node, UacAdjustment::class.java)
        }

        val contents = (jsonTree.get("result").get("contents") as? ArrayNode)?.mapNotNull { content ->
            val node = content as ObjectNode
            replaceEnumValueNode(node, "media_type")
            node.set("type", node.remove("media_type"))

            node.remove("orig_size")?.apply {
                node.set("iw", get("width"))
                node.set("ih", get("height"))
            }
            node.remove("content_size")?.apply {
                node.set("tw", get("width"))
                node.set("th", get("height"))
            }
            node.remove("orig_creative_size")?.apply {
                node.set("ow", get("width"))
                node.set("oh", get("height"))
            }

            val meta = node.get("meta") as ObjectNode

            // Для video и html5 формат данных в meta должен совпадать,
            // а для image надо переложить, так как использован proto из GrUT'а
            (meta.remove("image") as? ObjectNode)?.let { imageMeta ->
                (imageMeta.get("avatars_image_meta") as? ObjectNode)?.let { avatarsImageMeta ->
                    val colorWiz = objectMapper.createObjectNode()
                    avatarsImageMeta.get("color_wiz_background")?.let {
                        colorWiz.put("ColorWizBack", uint64ToColorString(it.textValue().toLong()))
                    }
                    avatarsImageMeta.get("color_wiz_button")?.let {
                        colorWiz.put("ColorWizButton", uint64ToColorString(it.textValue().toLong()))
                    }
                    avatarsImageMeta.get("color_wiz_button_text")?.let {
                        colorWiz.put("ColorWizButtonText", uint64ToColorString(it.textValue().toLong()))
                    }
                    avatarsImageMeta.get("color_wiz_text")?.let {
                        colorWiz.put("ColorWizText", uint64ToColorString(it.textValue().toLong()))
                    }
                    meta.set("ColorWiz", colorWiz)

                    imageMeta.remove("direct_image_hash")?.let {
                        meta.set("direct_image_hash", it)
                    }

                    (imageMeta.get("direct_mds_meta") as? ObjectNode)?.let { directMdsMeta ->
                        for (item in directMdsMeta) {
                            val itemObj = item as ObjectNode
                            itemObj.remove("size")?.let {
                                itemObj.set("width", it.get("width"))
                                itemObj.set("height", it.get("height"))
                            }
                            itemObj.remove("smart_centers")?.let { itemObj.set("smart-centers", it) }
                        }
                        meta.set("direct_mds_meta", directMdsMeta)
                    }

                    (avatarsImageMeta.get("formats") as? ArrayNode)?.let { formats ->
                        for (format in formats) {
                            val formatObj = format as ObjectNode
                            val formatName = formatObj.remove("format_name").textValue()
                            formatObj.remove("smart_centers")?.let { formatObj.set("smart-centers", it) }
                            formatObj.remove("smart_center")?.let { formatObj.set("smart-center", it) }
                            meta.set(formatName, formatObj)
                        }
                    }
                }
            }
            for (variant in listOf("video", "html5")) {
                (meta.remove(variant) as? ObjectNode)?.let {
                    for ((k, v) in it.fields()) {
                        if (k == "creative_id" || k == "preset_id") {
                            // Protobuf при переводе в json представляет 64-битные числа как строки, из-за чего возникает дифф
                            try {
                                meta.put(k, v.textValue().toInt())
                            } catch (_: NumberFormatException) {
                                meta.put(k, v.textValue().toLong())
                            }
                        } else {
                            meta.set(k, v)
                        }
                    }
                }
            }
            treeToValueSafe(node, Content::class.java)
        } ?: emptyList()

        val uacCampaign = UacCampaign(
            id = campaignMeta.id.toIdString(),
            showsKt = result.shows,
            sumKt = UacYdbUtils.moneyFromDb(result.sum).setScale(MONEY_CENT_SCALE, RoundingMode.DOWN),
            advType = campaignMeta.campaignType.toAdvType()!!,
            displayName = campaignSpec.campaignData.name,
            texts = result.textsList,
            titles = result.titlesList,
            regions = brief.regionsList,
            regionNames = result.regionNamesList,
            minusRegions = brief.minusRegionsList,
            minusRegionNames = result.minusRegionNamesList,
            trackingUrl = brief.targetHref.trackingUrl,
            impressionUrl = brief.targetHref.impressionUrl,
            href = brief.targetHref.href,
            faviconLink = result.faviconLink,
            targetId = brief.strategy.mobileAppCampaignTargetType.toTargetType(),
            cpa = UacYdbUtils.moneyFromDb(brief.strategy.cpa),
            weekLimit = UacYdbUtils.moneyFromDb(brief.strategy.weekLimit),
            createdTime = DateTimeUtils.fromEpochSeconds(campaignMeta.creationTime / 1_000_000),
            updatedTime = DateTimeUtils.fromEpochSeconds(campaignSpec.updateTime.toLong()),
            startedTime = DateTimeUtils.fromEpochSeconds(campaignSpec.startTime.toLong()),
            status = UacEnumMappers.fromProtoStatus(result.status)!!,
            isStatusObsolete = result.isStatusObsolete,
            extStatus = null,
            targetStatus = brief.targetStatus.toTargetStatus()!!,
            directId = campaignMeta.id,
            rejectReasons = null,
            contentFlags = brief.contentFlags.toContentFlags(),
            stateReasons = result.stateReasonsList.map { convertEnumValue(it.name, lowercase = false) },
            limitPeriod = brief.strategy.limitPeriod.toLimitPeriodType() ?: LimitPeriodType.WEEK,
            skadNetworkEnabled = brief.skadNetworkEnabled,
            adultContentEnabled = brief.adultContentEnabled,
            keywords = brief.keywordsList,
            minusKeywords = result.minusKeywordsList,
            deviceTypes = brief.deviceTypeList.toDeviceTypes(),
            inventoryTypes = brief.inventoryTypeList.toInventoryTypes(),
            counters = campaignSpec.campaignData.countersList,
            permalinkId = brief.permalinkId,
            phoneId = brief.phoneId,
            calltrackingSettingsId = brief.calltrackingSettingsId,
            timeTarget = if (brief.hasTimeTarget()) brief.timeTarget.toTimeTarget() else null,
            strategy = if (brief.hasStrategy()) brief.strategy.toUacStrategy() else null,
            videosAreNonSkippable = brief.videosAreNonSkippable,
            zenPublisherId = brief.zenPublisherId,
            brandSurveyId = brief.cpmData.brandSurveyId,
            brandSurveyName = null,
            showsFrequencyLimit = if (brief.cpmData.hasShowsFrequencyLimit()) {
                brief.cpmData.showsFrequencyLimit.toShowsFrequencyLimit()
            } else {
                null
            },
            strategyPlatform = if (brief.strategy.hasPlatform()) {
                brief.strategy.platform.toUacStrategyPlatform()
            } else {
                null
            },
            isEcom = brief.hasEcom(),
            crr = brief.strategy.crr,
            feedId = brief.ecom.feedId,
            feedFilters = brief.ecom.feedFiltersList.map { it.toUacFeedFilter() },
            showOfferStats = result.showOfferStats,
            trackingParams = brief.trackingParams,
            cpmAssets = brief.cpmData.cpmAssetsMap.mapValues { it.value.toCpmAsset() },
            campaignMeasurers = brief.cpmData.campaignMeasurersList.map {
                UacCampaignMeasurer(
                    UacCampaignMeasurerSystem.valueOf(it.measurerType),
                    it.params
                )
            },
            uacBrandsafety = if (brief.cpmData.hasBrandSafety()) {
                brief.cpmData.brandSafety.toUacBrandSafety()
            } else null,
            uacDisabledPlaces = if (brief.cpmData.hasDisabledPlaces()) {
                brief.cpmData.disabledPlaces.toUacDisabledPlaces()
            } else null,
            isRecommendationsManagementEnabled = brief.recommendationsManagementEnabled,
            isPriceRecommendationsManagementEnabled = brief.priceRecommendationsManagementEnabled,
            relevanceMatchCategories = BaseUacCampaignWebService.getRelevanceMatchCategories(
                brief.relevanceMatch.toRelevanceMatch()
            ),
            showTitleAndBody = brief.showTitleAndBody,
            altAppStores = result.altAppStoresList
                .mapNotNull { UacEnumMappers.fromProtoAppStore(it) }
                .map { it.fromCoreType() }
                .toSet(),
            bizLandingId = campaignSpec.bizLandingId,
            bid = result.minBid,
            accessKt = accessKt,
            agencyInfoKt = campaignAgencyInfoKt,
            managerInfoKt = managerInfoKt,
            appInfo = appInfo,
            hyperGeo = hyperGeo,
            socdem = if (brief.hasSocdem()) {
                brief.socdem.toSocdem()
            } else {
                null
            },
            goals = brief.strategy.goalsList.toUacGoals(),
            sitelinks = sitelinks,
            brandSurveyStatus = brandSurveyStatus,
            adjustments = adjustments,
            retargetingCondition = if (result.hasRetargetingCondition()) {
                result.retargetingCondition.toRetargetingCondition()
            } else {
                null
            },
            contents = contents,
            searchLift = if (brief.cpmData.hasSearchLift()) {
                brief.cpmData.searchLift.toUacSearchLift()
            } else {
                null
            },
        )

        return UacCampaignResponse(from.reqId).withResult(uacCampaign)
    }
}
