package ru.yandex.direct.web.entity.uac.converter.proto

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.node.ArrayNode
import com.fasterxml.jackson.databind.node.ObjectNode
import com.fasterxml.jackson.databind.node.TextNode
import org.slf4j.Logger
import ru.yandex.direct.web.entity.uac.model.UacProtoResponse

/**
 * Маппинг ответа читающих ручек из протобуфного формата в обычный. Используется в 2 целях:
 * 1) как документация изменений в схеме для фронта
 * 2) для проверки обратной совместимости форматов
 *
 * Код задумывался как временный и предназначенный только для тестирования
 */

abstract class UacProtoResponseMapper<From : UacProtoResponse<*>, To>(
    protected val objectMapper: ObjectMapper,
) {
    abstract fun logger(): Logger
    abstract fun convert(from: From): To

    fun <T> treeToValueSafe(jsonNode: JsonNode, cls: Class<T>): T? {
        return try {
            objectMapper.treeToValue(jsonNode, cls)
        } catch (e: Exception) {
            logger().warn("treeToValue failed. {}", e.message)
            null
        }
    }

    companion object {
        fun convertEnumValue(s: String, lowercase: Boolean = true): String {
            val result = s.replace(Regex("^[A-Z]+_"), "")
            return if (lowercase) result.lowercase() else result
        }

        // Workaround для repeated-полей в proto, соответствующих not-nullable коллекциям в data class'ах
        fun addEmptyArrayIfMissing(node: ObjectNode, key: String, objectMapper: ObjectMapper) {
            if (node[key] == null) {
                node[key] = objectMapper.createArrayNode()
            }
        }

        fun convertEnumArray(node: JsonNode?, lowercase: Boolean = true) {
            (node as? ArrayNode)?.let {
                for (i in IntRange(0, it.size() - 1)) {
                    it.set(i, TextNode.valueOf(convertEnumValue(it.get(i).asText(), lowercase)))
                }
            }
        }

        fun replaceEnumValueNode(node: JsonNode?, key: String, lowercase: Boolean = true) {
            (node as? ObjectNode)?.remove(key)?.let {
                node.put(key, convertEnumValue(it.textValue(), lowercase))
            }
        }
    }
}
