package ru.yandex.direct.web.entity.uac.service

import ru.yandex.direct.core.aggregatedstatuses.AggregatedStatusesService
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RecalculationDepthEnum
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService
import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.campaign.service.CampaignBudgetReachService
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter
import ru.yandex.direct.core.entity.feed.model.StatusMBISynced
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService
import ru.yandex.direct.core.entity.uac.converter.UacBidModifiersConverter.toUacAdjustments
import ru.yandex.direct.core.entity.uac.converter.UacCampaignConverter.toCampaignStatuses
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.buildRetargetingCondition
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.getRetargetingCondition
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toEBriefStatus
import ru.yandex.direct.core.entity.uac.converter.UacRegionNamesConverter.regionIdsToNames
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.AltAppStore.Companion.fromCoreType
import ru.yandex.direct.core.entity.uac.model.CampaignStatuses
import ru.yandex.direct.core.entity.uac.model.LimitPeriodType
import ru.yandex.direct.core.entity.uac.model.UacAdjustment
import ru.yandex.direct.core.entity.uac.model.relevance_match.UacRelevanceMatch
import ru.yandex.direct.core.entity.uac.model.relevance_match.UacRelevanceMatchCategories
import ru.yandex.direct.core.entity.uac.model.relevance_match.UacRelevanceMatchCategory
import ru.yandex.direct.core.entity.uac.model.relevance_match.toUacRelevanceMatchCategoryItem
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.moneyToDb
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.FetchedContents
import ru.yandex.direct.core.entity.uac.service.RmpCampaignService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.grid.model.campaign.GdiCampaign
import ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.extractCampaignSum
import ru.yandex.direct.grid.processing.model.client.GdClientInfo
import ru.yandex.direct.grid.processing.service.campaign.CampaignAgencyManagerInfoHelper
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService
import ru.yandex.direct.grid.processing.service.client.ClientDataService
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.utils.mapToSet
import ru.yandex.direct.web.entity.uac.converter.UacCampaignConverter
import ru.yandex.direct.web.entity.uac.converter.proto.UacCampaignProtoConverter.toProto
import ru.yandex.direct.web.entity.uac.converter.proto.UacContentProtoConverter.toProto
import ru.yandex.direct.web.entity.uac.converter.proto.enummappers.UacEnumMappers.Companion.toProtoAltAppStore
import ru.yandex.direct.web.entity.uac.converter.proto.enummappers.UacEnumMappers.Companion.toProtoCampaignType
import ru.yandex.direct.web.entity.uac.converter.proto.enummappers.UacEnumMappers.Companion.toProtoSelfStatusReason
import ru.yandex.direct.web.entity.uac.converter.proto.enummappers.UacEnumMappers.Companion.toProtoStatus
import ru.yandex.direct.web.entity.uac.model.UacCampaign
import ru.yandex.direct.web.entity.uac.model.UacCampaignAccess
import ru.yandex.direct.web.entity.uac.model.UacCampaignAgencyInfo
import ru.yandex.direct.web.entity.uac.model.UacCampaignManagerInfo
import ru.yandex.direct.web.entity.uac.model.UacContactPhone
import ru.yandex.direct.web.proto.api.uac.GetCampaignResponse.TGetCampaignResponse
import ru.yandex.grut.objects.proto.Campaign.ECampaignTypeOld
import ru.yandex.grut.objects.proto.client.Schema.TCampaign
import java.math.BigDecimal
import java.time.Instant
import java.time.LocalDateTime.now

abstract class BaseUacCampaignWebService(
    private val campaignAgencyManagerInfoHelper: CampaignAgencyManagerInfoHelper,
    private val campaignInfoService: CampaignInfoService,
    private val clientGeoService: ClientGeoService,
    private val shardHelper: ShardHelper,
    private val campaignRepository: CampaignRepository,
    private val hyperGeoService: HyperGeoService,
    private val rmpCampaignService: RmpCampaignService,
    private val uacMobileAppService: UacMobileAppService,
    private val aggregatedStatusesService: AggregatedStatusesService,
    private val campaignBudgetReachService: CampaignBudgetReachService,
    private val bidModifierService: BidModifierService,
    private val uacRetargetingConditionService: UacRetargetingConditionService,
    private val clientDataService: ClientDataService,
    private val feedService: FeedService
) {

    fun fillCampaign(
        operator: User,
        subjectUser: User,
        campaign: UacYdbCampaign,
        directCampaignId: Long,
        campaignStatuses: CampaignStatuses,
    ): UacCampaign {
        return fillCampaign(
            operator, subjectUser, campaign, directCampaignId, campaignStatuses,
            getMinBannerIdForCampaign(campaign.id)
        )
    }

    /**
     * @param operator описание оператора
     * @param subjectUser описание пользователя
     * @param campaign заявочная кампания
     * @param directCampaignId id директовой кампании
     * @param bid id баннера кампании
     */
    fun fillCampaign(
        operator: User,
        subjectUser: User,
        campaign: UacYdbCampaign,
        directCampaignId: Long,
        campaignStatuses: CampaignStatuses,
        bid: Long?,
    ): UacCampaign {
        val appInfo = uacMobileAppService.getAppInfo(campaign.appId)

        val regionIds = campaign.regions
        val minusRegionIds = campaign.minusRegions
        val regionNames = regionIdsToNames(operator.clientId, regionIds)
        val minusRegionNames = regionIdsToNames(operator.clientId, minusRegionIds)

        val shard = shardHelper.getShardByCampaignId(directCampaignId)
        val campaignSimple = getCampaignSimple(shard, directCampaignId)

        val fetchedContents = fetchContents(CampaignContainer.Kt(campaign))

        // Не хотим показывать статус "Обрабатывается" во время первой генерации баннеров - хотим статус "На модерации"
        val isCampStatusObsolete = if (bid == null && campaign.briefSynced == false) {
            false
        } else {
            (campaignStatuses.isCampaignAggrStatusObsolete || !(campaign.briefSynced ?: true))
        }

        val clientInfo = getClientInfo(operator, subjectUser)
        val gdiCampaignMap = campaignInfoService
            // Используем данный метод, поскольку он не использует GridGraphQLContext
            .getCampaignsByIdsAndAllWallets(setOf(directCampaignId), subjectUser, operator, clientInfo)

        val gdiCampaign = gdiCampaignMap[directCampaignId]

        val brandSurveyStatus = getBrandSurveyStatus(shard, directCampaignId, subjectUser, campaign.brandSurveyId)

        return UacCampaign(
            id = campaign.id,
            accessKt = getCampaignAccess(campaignSimple, directCampaignId, clientInfo, operator, gdiCampaignMap),
            agencyInfoKt = getAgencyInfo(campaignSimple),
            managerInfoKt = getManagerInfo(campaignSimple),
            showsKt = gdiCampaign?.shows ?: 0,
            sumKt = gdiCampaign?.let { extractCampaignSum(it) } ?: BigDecimal.ZERO,
            advType = campaign.advType,
            displayName = campaign.name,
            appInfo = appInfo,
            contents = fetchedContents.mediaContents,
            texts = fetchedContents.texts,
            titles = fetchedContents.titles,
            regions = regionIds,
            regionNames = regionNames,
            minusRegions = minusRegionIds,
            minusRegionNames = minusRegionNames,
            trackingUrl = campaign.trackingUrl,
            impressionUrl = campaign.impressionUrl,
            href = campaign.storeUrl,
            faviconLink = UacCampaignConverter.makeFaviconLink(campaign.advType, campaign.storeUrl),
            targetId = campaign.targetId,
            cpa = campaign.cpa,
            weekLimit = campaign.weekLimit,
            createdTime = campaign.createdAt,
            updatedTime = campaign.updatedAt,
            startedTime = campaign.startedAt,
            status = campaignStatuses.status,
            isStatusObsolete = isCampStatusObsolete,
            extStatus = null,
            targetStatus = campaignStatuses.targetStatus,
            directId = directCampaignId,
            rejectReasons = null, //not used field
            contentFlags = campaign.contentFlags,
            stateReasons = campaignStatuses.stateReasons?.map { it.name },
            limitPeriod = campaign.options?.limitPeriod ?: LimitPeriodType.WEEK,
            skadNetworkEnabled = campaign.skadNetworkEnabled,
            adultContentEnabled = campaign.adultContentEnabled,
            hyperGeo = campaign.hyperGeoId?.let { hyperGeoService.getHyperGeoById(subjectUser.clientId, it) },
            keywords = campaign.keywords,
            minusKeywords = if (campaign.advType == AdvType.TEXT && campaignSimple != null) {
                campaignSimple.minusKeywords
            } else {
                campaign.minusKeywords
            },
            socdem = campaign.socdem,
            deviceTypes = campaign.deviceTypes,
            inventoryTypes = campaign.inventoryTypes,
            goals = campaign.goals,
            counters = campaign.counters,
            permalinkId = campaign.permalinkId,
            phoneId = campaign.phoneId,
            calltrackingSettingsId = campaign.calltrackingSettingsId,
            sitelinks = fetchedContents.sitelinks,
            timeTarget = campaign.timeTarget,
            bid = bid,
            strategy = campaign.strategy,
            retargetingCondition = uacRetargetingConditionService.fillRetargetingCondition(
                shard, campaignSimple, campaign.retargetingCondition
            ),
            videosAreNonSkippable = campaign.videosAreNonSkippable,
            zenPublisherId = campaign.zenPublisherId,
            brandSurveyId = campaign.brandSurveyId,
            brandSurveyName = null,
            showsFrequencyLimit = campaign.showsFrequencyLimit,
            strategyPlatform = campaign.strategyPlatform,
            brandSurveyStatus = brandSurveyStatus,
            adjustments = getAdjustments(subjectUser, operator, directCampaignId, campaign.advType),
            isEcom = campaign.isEcom,
            crr = campaign.crr,
            feedId = campaign.feedId,
            feedFilters = campaign.feedFilters,
            showOfferStats = shouldShowOfferStats(subjectUser, campaign.feedId),
            trackingParams = campaign.trackingParams,
            cpmAssets = campaign.cpmAssets,
            campaignMeasurers = campaign.campaignMeasurers,
            uacBrandsafety = campaign.uacBrandsafety,
            uacDisabledPlaces = campaign.uacDisabledPlaces,
            isRecommendationsManagementEnabled = campaign.recommendationsManagementEnabled,
            isPriceRecommendationsManagementEnabled = campaign.priceRecommendationsManagementEnabled,
            relevanceMatchCategories = getRelevanceMatchCategories(campaign.relevanceMatch),
            showTitleAndBody = campaign.showTitleAndBody,
            altAppStores = gdiCampaign?.alternativeAppStores?.mapToSet { it.fromCoreType() },
            bizLandingId = campaign.bizLandingId,
            searchLift = campaign.searchLift,
        )
    }

    fun fillCampaignProto(
        operator: User,
        subjectUser: User,
        campaign: TCampaign,
        clientId: ClientId,
    ): TGetCampaignResponse {
        val bid = getMinBannerIdForCampaign(campaign.meta.id.toIdString())
        val campaignStatuses = recalcStatuses(
            clientId, campaign.meta.id, campaign.spec.startTime == 0, bid = bid
        )!! // Null может быть только при отсутствии кампании в директе (а она создаётся при создании заявки)
        val brief = campaign.spec.campaignBrief
        val appInfo = brief.appId.takeIf { it.isNotEmpty() } ?.let { uacMobileAppService.getAppInfo(it) }

        val regionIds = brief.regionsList
        val minusRegionIds = brief.minusRegionsList

        val cid = campaign.meta.id
        val shard = shardHelper.getShardByCampaignId(cid)
        val campaignSimple = getCampaignSimple(shard, cid)

        // Не хотим показывать статус "Обрабатывается" во время первой генерации баннеров - хотим статус "На модерации"
        val synced = if (brief.hasBriefSynced()) brief.briefSynced else null
        val isCampStatusObsolete = if (bid == null && synced == false) {
            false
        } else {
            campaignStatuses.isCampaignAggrStatusObsolete || !(synced ?: true)
        }

        val clientInfo = getClientInfo(operator, subjectUser)
        val gdiCampaignMap = campaignInfoService
            // Используем данный метод, поскольку он не использует GridGraphQLContext
            .getCampaignsByIdsAndAllWallets(setOf(cid), subjectUser, operator, clientInfo)
        val gdiCampaign = gdiCampaignMap[cid]
        val brandSurveyStatus = getBrandSurveyStatus(shard, cid, subjectUser, brief.cpmData.brandSurveyId)
        val campaignAccess = getCampaignAccess(campaignSimple, cid, clientInfo, operator, gdiCampaignMap)
        val agencyInfo = getAgencyInfo(campaignSimple)
        val managerInfo = getManagerInfo(campaignSimple)
        val hyperGeo = if (brief.hypergeoId != 0L) {
            hyperGeoService.getHyperGeoById(subjectUser.clientId, brief.hypergeoId)
        } else {
            null
        }
        val faviconLink = UacCampaignConverter.makeFaviconLink(campaign.meta.campaignType, brief.targetHref.href)
        val adjustments = getAdjustments(subjectUser, operator, cid, campaign.meta.campaignType) ?: emptyList()
        val fetchedContents = fetchContents(CampaignContainer.Proto(campaign))

        val retargetingCondition = uacRetargetingConditionService.fillRetargetingCondition(
            shard, campaignSimple, getRetargetingCondition(campaign.spec.campaignBrief)
        )

        return TGetCampaignResponse.newBuilder().apply {
            campaignBuilder.apply {
                // Прокидываем заявку как есть
                mergeFrom(campaign)

                specBuilder.campaignBriefBuilder.apply {
                    targetStatus = campaignStatuses.targetStatus.toEBriefStatus()
                }
            }

            // Заполняем всё остальное из директовых моделей
            addAllRegionNames(regionIdsToNames(operator.clientId, regionIds))
            addAllMinusRegionNames(regionIdsToNames(operator.clientId, minusRegionIds))
            status = toProtoStatus(campaignStatuses.status)
            isStatusObsolete = isCampStatusObsolete
            if (campaignStatuses.stateReasons != null) {
                addAllStateReasons(campaignStatuses.stateReasons!!.map { toProtoSelfStatusReason(it) })
            }
            addAllMinusKeywords(
                if (campaign.meta.campaignType == ECampaignTypeOld.CTO_TEXT && campaignSimple != null) {
                    campaignSimple.minusKeywords
                } else {
                    brief.minusKeywordsList
                }
            )
            access = toProto(campaignAccess)
            if (agencyInfo != null) {
                this.agencyInfo = toProto(agencyInfo)
            }
            if (managerInfo != null) {
                this.managerInfo = toProto(managerInfo)
            }
            shows = gdiCampaign?.shows ?: 0
            if (gdiCampaign != null) {
                sum = moneyToDb(extractCampaignSum(gdiCampaign))
            }
            if (appInfo != null) {
                this.appInfo = toProto(appInfo)
            }
            if (faviconLink != null) {
                this.faviconLink = faviconLink
            }
            if (hyperGeo != null) {
                this.hyperGeo = toProto(hyperGeo)
            }
            if (brandSurveyStatus != null) {
                this.brandSurveyStatus = toProto(brandSurveyStatus)
            }
            showOfferStats = shouldShowOfferStats(subjectUser, brief.ecom.feedId)
            if (gdiCampaign != null && gdiCampaign.alternativeAppStores != null) {
                addAllAltAppStores(gdiCampaign.alternativeAppStores.map { toProtoAltAppStore(it) })
            }
            addAllAdjustments(adjustments.map { toProto(it) })
            if (fetchedContents.texts != null) {
                addAllTexts(fetchedContents.texts)
            }
            if (fetchedContents.titles != null) {
                addAllTitles(fetchedContents.titles)
            }
            if (fetchedContents.sitelinks != null) {
                addAllSitelinks(fetchedContents.sitelinks!!.map { toProto(it) })
            }
            addAllContents(fetchedContents.mediaContents.map { toProto(it) })
            if (bid != null) {
                minBid = bid
            }
            if (retargetingCondition != null) {
                this.retargetingCondition = buildRetargetingCondition(retargetingCondition)
            }
        }.build()
    }

    fun recalcStatuses(
        clientId: ClientId,
        directCampaignId: Long,
        isDraft: Boolean,
        withRecalculatingAggrStatuses: Boolean = false,
        bid: Long? = null
    ): CampaignStatuses? {
        if (withRecalculatingAggrStatuses) {
            Trace.current().profile("aggregatedStatuses:fullyRecalculate").use {
                aggregatedStatusesService.fullyRecalculateStatuses(
                    shardHelper.getShardByCampaignId(directCampaignId), now(),
                    setOf(directCampaignId), RecalculationDepthEnum.ALL
                )
            }
        }
        val states = rmpCampaignService.getCampaignStates(
            clientId,
            directCampaignId,
            true,
            bid
        ) ?: return null
        return states.toCampaignStatuses(isDraft)
    }

    fun regionIdsToNames(operatorClientId: ClientId, regionIds: List<Long>?): List<String>? {
        if (regionIds == null) {
            return null
        }

        val geoTree = clientGeoService.getClientTranslocalGeoTree(operatorClientId)
        return regionIdsToNames(geoTree, regionIds)
    }

    fun getAdjustments(
        subjectUser: User,
        operator: User,
        directCampaignId: Long,
        advType: AdvType,
    ): List<UacAdjustment>? {
        return getAdjustments(subjectUser, operator, directCampaignId, toProtoCampaignType(advType))
    }

    fun getAdjustments(
        subjectUser: User,
        operator: User,
        directCampaignId: Long,
        advType: ECampaignTypeOld,
    ): List<UacAdjustment>? {
        val bidModifiers = bidModifierService.getByCampaignIds(
            subjectUser.clientId, listOf(directCampaignId),
            setOf(
                BidModifierType.DEMOGRAPHY_MULTIPLIER,
                BidModifierType.GEO_MULTIPLIER,
                BidModifierType.RETARGETING_MULTIPLIER
            ),
            setOf(BidModifierLevel.CAMPAIGN), operator.uid
        )
        val geoTree = clientGeoService.getClientTranslocalGeoTree(subjectUser.clientId)
        return if (advType == ECampaignTypeOld.CTO_MOBILE_APP) {
            toUacAdjustments(bidModifiers, geoTree)
        } else {
            null
        }
    }

    fun getCampaignSimple(
        shard: Int,
        cid: Long,
    ): CampaignSimple? = campaignRepository
        .getCampaignsSimple(shard, setOf(cid))[cid]

    fun getCampaignAccess(
        campaignSimple: CampaignSimple?,
        cid: Long,
        clientInfo: GdClientInfo,
        operator: User,
        gdiCampaignMap: Map<Long, GdiCampaign>
    ): UacCampaignAccess {
        return (campaignSimple?.clientId?.let {
            campaignInfoService
                // Используем данный метод, поскольку он не использует GridGraphQLContext
                .getCampaignsAccess(
                    setOf(cid),
                    gdiCampaignMap,
                    operator,
                    clientInfo,
                    Instant.now()
                )
                .get(cid)
        } ?: CampaignInfoService.DEFAULT_ACCESS).let {
            UacCampaignAccess.fromGdCampaignAccess(it)
        }
    }

    fun getCampaignAccess(
        campaignSimple: CampaignSimple?,
        cid: Long,
        subjectUser: User,
        operator: User,
    ): UacCampaignAccess {
        val clientInfo = clientDataService
            .getClientInfo(operator, setOf(subjectUser.clientId.asLong()))
            .stream()
            .findFirst()
            .orElseThrow {
                IllegalArgumentException(
                    "cant find client"
                )
            }
        val gdiCampaignMap = campaignInfoService.getCampaignsByIdsAndAllWallets(
            setOf(cid),
            subjectUser,
            operator,
            clientInfo
        )
        return getCampaignAccess(campaignSimple, cid, clientInfo, operator, gdiCampaignMap)
    }

    fun getAgencyInfo(
        campaignSimple: CampaignSimple?,
    ): UacCampaignAgencyInfo? {
        return campaignSimple?.agencyUserId?.let {
            campaignAgencyManagerInfoHelper
                .getCampaignAgencyInfoByUid(setOf(it))
                .get(it)
        }?.let {
            UacCampaignAgencyInfo(
                email = it.email,
                name = it.name,
                phone = it.phone,
                representativeName = it.representativeName,
                showAgencyContacts = it.showAgencyContacts
            )
        }
    }

    fun getManagerInfo(
        campaignSimple: CampaignSimple?,
    ): UacCampaignManagerInfo? {
        return campaignSimple?.managerUserId?.let {
            campaignAgencyManagerInfoHelper
                .getCampaignManagerInfoByUid(setOf(it))
                .get(it)
        }?.let {
            UacCampaignManagerInfo(
                contactInfo = it.contactInfo?.map { contactPhone ->
                    UacContactPhone(
                        phone = contactPhone.phone,
                        extension = contactPhone.extension,
                    )
                },
                email = it.email,
                name = it.name
            )
        }
    }

    companion object {
        /**
         * Возвращает категории автотаргетинга с флагом selected=true если:
         * - relevanceMatch == null
         * - ИЛИ !relevanceMatch.active
         * - ИЛИ категория есть в заявке (relevanceMatch.categories)
         */
        fun getRelevanceMatchCategories(
            relevanceMatch: UacRelevanceMatch?
        ): UacRelevanceMatchCategories {
            val active = relevanceMatch?.active ?: false
            val uacRelevanceMatchCategory =
                if (active && relevanceMatch != null) relevanceMatch.categories else emptySet()
            val categories = UacRelevanceMatchCategory.values()
                .map {
                    toUacRelevanceMatchCategoryItem(
                        it, uacRelevanceMatchCategory.isEmpty() || uacRelevanceMatchCategory.contains(it)
                    )
                }
            return UacRelevanceMatchCategories(
                active = active,
                categories = categories,
            )
        }
    }

    fun shouldShowOfferStats(
        subjectUser: User,
        feedId: Long?
    ): Boolean {
        if (feedId == null || feedId == 0L) {
            return false
        }
        val feeds = feedService.getFeedsSimple(subjectUser.clientId, FeedQueryFilter.newBuilder().run {
            withFeedIds(listOf(feedId))
            build()
        })
        return feeds.getOrNull(0)?.statusMbiSynced == StatusMBISynced.YES
    }

    private fun getClientInfo(operator: User, subjectUser: User): GdClientInfo {
        return clientDataService
            .getClientInfo(operator, setOf(subjectUser.clientId.asLong()))
            .stream()
            .findFirst()
            .orElseThrow {
                IllegalArgumentException(
                    "cant find client"
                )
            }
    }

    private fun getBrandSurveyStatus(
        shard: Int,
        cid: Long,
        subjectUser: User,
        brandSurveyId: String?
    ): BrandSurveyStatus? {
        if (brandSurveyId.isNullOrEmpty()) {
            return null
        }
        return campaignBudgetReachService.getBrandStatusForCampaigns(
            shard,
            subjectUser.clientId,
            mapOf(cid to brandSurveyId)
        )[cid]
    }

    abstract fun getMinBannerIdForCampaign(id: String): Long?
    abstract fun fetchContents(campaign: CampaignContainer): FetchedContents
}

sealed class CampaignContainer {
    class Proto(val value: TCampaign) : CampaignContainer()
    class Kt(val value: UacYdbCampaign) : CampaignContainer()
}
