package ru.yandex.direct.web.entity.uac.service

import java.util.Locale
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.canvas.client.model.html5.Html5Tag
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.CreativeType
import ru.yandex.direct.core.entity.uac.model.FileData
import ru.yandex.direct.core.entity.uac.model.UpdateVideoContentJobParams
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.defect.FileDefects
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.web.entity.uac.model.CreateContentRequest
import ru.yandex.direct.web.entity.uac.regionsEmptyResponse
import ru.yandex.direct.web.entity.uac.toResponse
import ru.yandex.direct.web.entity.uac.trackingUrlEmptyResponse
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService

abstract class BaseUacContentWebService(
    private val uacImageContentUploader: UacImageContentUploader,
    private val uacVideoContentUploader: UacVideoContentUploader,
    private val uacHtml5ContentUploader: UacHtml5ContentUploader,
    private val validationResultConversionService: ValidationResultConversionService,
    private val dbQueueRepository: DbQueueRepository,
    private val shardHelper: ShardHelper,
) {
    abstract fun checkCampaignIsCompleteForUpdateStatus(campaign: UacYdbCampaign): ResponseEntity<Any>?

    protected fun checkCampaignIsCompleteCommon(campaign: UacYdbCampaign): ResponseEntity<Any>? {
        if (campaign.advType == AdvType.MOBILE_CONTENT && campaign.regions.isNullOrEmpty()) {
            return regionsEmptyResponse()
        }

        if (campaign.advType == AdvType.MOBILE_CONTENT && campaign.trackingUrl.isNullOrEmpty()) {
            return trackingUrlEmptyResponse()
        }

        return null
    }

    /**
     * NB! в этом методе не создается транзакция GrUT, так как в ручке есть походы в canvas и avatars, и пишущий запрос всего один
     * При добавлении еще пишущего запроса, нужно сгруппировать их в одну транзакцию(НО без походов по сети в другие сервисы)
     */
    fun createContent(
        operator: User,
        subjectUser: User,
        creativeType: CreativeType,
        multipartFile: MultipartFile?,
        createContentRequest: CreateContentRequest?,
        accountId: String,
        locale: Locale?,
        html5Tag: Html5Tag
    ): ResponseEntity<Any> {
        val contentResult = if (createContentRequest != null)
            uploadContentFromRequest(subjectUser, accountId, createContentRequest, creativeType, locale)
        else uploadContentFromFile(subjectUser, accountId, multipartFile!!, creativeType, locale, html5Tag)

        return if (!contentResult.isSuccessful) {
            ResponseEntity(
                validationResultConversionService.buildValidationResponse(
                    contentResult
                ),
                HttpStatus.BAD_REQUEST,
            )
        } else {
            val content = contentResult.result!!
            insertContent(content)
            if (content.type == ru.yandex.direct.core.entity.uac.model.MediaType.VIDEO && needToUpdateVideoContent(
                    content
                )
            ) {
                dbQueueRepository.insertJob(
                    shardHelper.getShardByClientId(subjectUser.clientId),
                    DbQueueJobTypes.UAC_UPDATE_VIDEO_CONTENT,
                    subjectUser.clientId,
                    operator.uid,
                    UpdateVideoContentJobParams(uacYdbContentId = content.id)
                )
            }
            ResponseEntity(fillContent(content)!!.toResponse(), HttpStatus.OK)
        }
    }

    abstract fun insertContent(content: UacYdbContent)
    abstract fun fillContent(ydbContent: UacYdbContent?): Content?

    abstract fun uploadContent(
        operator: User,
        subjectUser: User,
        creativeType: CreativeType,
        multipartFile: MultipartFile?,
        createContentRequest: CreateContentRequest?,
        locale: Locale?
    ): Result<UacYdbContent>

    protected fun uploadContentFromFile(
        subjectUser: User, accountId: String,
        multipartFile: MultipartFile, creativeType: CreativeType, locale: Locale?,
        html5Tag: Html5Tag?
    ): Result<UacYdbContent> {
        val contentType = multipartFile.contentType ?: return Result.broken(
            ValidationResult.failed(
                null,
                FileDefects.noContentType()
            )
        )
        val fileData = FileData(name = multipartFile.originalFilename, data = multipartFile.bytes)
        if (contentType.contains("image")) {
            return if (creativeType == CreativeType.HTML5) {
                uacHtml5ContentUploader.upload(subjectUser, accountId, fileData, creativeType, locale, html5Tag)
            } else {
                uacImageContentUploader.uploadFromFile(subjectUser, accountId, fileData, creativeType)
            }
        }
        if (contentType.contains("video")) {
            return uacVideoContentUploader.uploadFromFile(subjectUser, accountId, fileData, creativeType, locale)
        }
        if (contentType.contains("zip")) {
            return uacHtml5ContentUploader.upload(subjectUser, accountId, fileData, creativeType, locale, html5Tag)
        }

        return Result.broken(ValidationResult.failed(contentType, FileDefects.fileMimeTypeIsNotSupported()))
    }

    protected fun uploadContentFromRequest(
        subjectUser: User, accountId: String,
        createContentRequest: CreateContentRequest, creativeType: CreativeType, locale: Locale?
    ): Result<UacYdbContent> {
        if (createContentRequest.thumb != null) {
            return uacImageContentUploader.uploadFromThumb(
                subjectUser, accountId, createContentRequest.thumb, creativeType
            )
        }
        if (
            createContentRequest.type == ru.yandex.direct.core.entity.uac.model.MediaType.IMAGE
            && createContentRequest.sourceUrl != null
        ) {
            return uacImageContentUploader.uploadFromUrl(
                subjectUser, accountId, createContentRequest.sourceUrl, creativeType
            )
        }
        if (createContentRequest.type == ru.yandex.direct.core.entity.uac.model.MediaType.VIDEO) {
            if (createContentRequest.mdsUrl != null) {
                return uacVideoContentUploader.uploadFromUrl(
                    subjectUser,
                    accountId,
                    createContentRequest.mdsUrl,
                    createContentRequest.sourceUrl,
                    creativeType,
                    locale
                )
            }
            if (createContentRequest.sourceUrl != null) {
                return uacVideoContentUploader.uploadFromUrl(
                    subjectUser,
                    accountId,
                    createContentRequest.sourceUrl,
                    createContentRequest.sourceUrl,
                    creativeType,
                    locale
                )
            }
        }
        return Result.broken(ValidationResult.failed(createContentRequest, CommonDefects.invalidValue()))
    }

    private fun needToUpdateVideoContent(uacYdbContent: UacYdbContent): Boolean {
        return (uacYdbContent.meta["status"] as? String) == "converting"
    }
}

