package ru.yandex.direct.web.entity.uac.service

import java.time.LocalDateTime
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.AssetContainer
import ru.yandex.direct.core.entity.uac.model.AssetLinkType
import ru.yandex.direct.core.entity.uac.model.CampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.Sitelink
import ru.yandex.direct.core.entity.uac.model.SitelinkAsset
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.generateRandomIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdString
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.service.GrutUacContentService
import ru.yandex.direct.core.grut.api.SitelinkAssetGrut
import ru.yandex.direct.core.grut.api.TextAssetGrut
import ru.yandex.direct.core.grut.api.TitleAssetGrut
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.web.entity.uac.model.PatchCampaignInternalRequest

data class CreatedAssets(
    val newTitleToId: Map<String, String>,
    val newTextToId: Map<String, String>,
    val newSitelinkToId: Map<Sitelink, String>,
)

@Service
class GrutUacAssetLinkUpdateService(
    private val grutContext: GrutContext,
    private val grutUacContentService: GrutUacContentService,
    private val grutApiService: GrutApiService,
) {

    fun getUpdatedAssetLinks(
        updatedUacYdbCampaign: UacYdbCampaign,
        request: PatchCampaignInternalRequest,
        clientId: ClientId,
        newAssetLinkIdsGenerationEnabled: Boolean = false,
    ): List<UacYdbCampaignContent> {
        val oldAssetContainer = grutUacContentService.getAssetContainer(updatedUacYdbCampaign)
        val newMediaAssetIds = request.contentIds ?: oldAssetContainer.mediaAssets.asSequence()
            .filter { !it.isRemoved }
            .map { it.asset.id }
            .toList()
        val nonMediaAssetIdToOrder = getNewNonMediaAssetIdToOrder(oldAssetContainer, request, clientId)
        val newAssetIds = nonMediaAssetIdToOrder.keys + newMediaAssetIds

        val notRestoreOldLinks = shouldNotRestoreOldLinks(updatedUacYdbCampaign, newAssetLinkIdsGenerationEnabled)
        val oldAssetLinks = updatedUacYdbCampaign.assetLinks!!
        val (actualAssetLinks, historyAssetLinks) = oldAssetLinks.partition {
            !notRestoreOldLinks || it.removedAt == null
        }

        val createdAssetIds = actualAssetLinks.asSequence()
            .filter { it.removedAt == null }
            .map { it.contentId!! }
            .toSet()
        val removedAssetIds = actualAssetLinks.asSequence()
            .filter { it.removedAt != null }
            .map { it.contentId!! }
            .filter { !createdAssetIds.contains(it) }
            .toSet()

        val updatedAssetLinks = updateAssetLinks(actualAssetLinks,
            removedAssetIds = removedAssetIds,
            createdAssetIds = createdAssetIds,
            newAssetIds = newAssetIds,
            updatedUacYdbCampaign,
            nonMediaAssetIdToOrder,
            newAssetLinkIdsGenerationEnabled)

        return updatedAssetLinks + historyAssetLinks
    }

    private fun getNewNonMediaAssetIdToOrder(
        oldAssetContainer: AssetContainer,
        request: PatchCampaignInternalRequest,
        clientId: ClientId,
    ): Map<String, Int> {
        val createdAbsentAssets = createAbsentNonMediaContents(oldAssetContainer, request, clientId)

        val oldTitleToId = oldAssetContainer.titles.associate { it.asset.title to it.asset.id }
        val titleToId = createdAbsentAssets.newTitleToId + oldTitleToId
        val titlesToSet = request.titles ?: oldAssetContainer.titles
            .asSequence()
            .filter { !it.isRemoved }
            .sortedBy { it.order }
            .map { it.asset.title }
            .toList()
        val titleIdsToUpdate = titlesToSet.mapIndexed { index, title -> titleToId[title]!! to index }

        val oldTextToId = oldAssetContainer.texts.associate { it.asset.text to it.asset.id }
        val textToId = createdAbsentAssets.newTextToId + oldTextToId
        val textsToSet = request.texts ?: oldAssetContainer.texts
            .asSequence()
            .filter { !it.isRemoved }
            .sortedBy { it.order }
            .map { it.asset.text }
            .toList()
        val textIdsToUpdate = textsToSet.mapIndexed { index, text -> textToId[text]!! to index }

        val oldSitelinkToId = oldAssetContainer.sitelinks.associate { it.asset.toSitelink() to it.asset.id }
        val sitelinkToId = createdAbsentAssets.newSitelinkToId + oldSitelinkToId
        val sitelinksToSet = request.sitelinks ?: oldAssetContainer.sitelinks
            .asSequence()
            .filter { !it.isRemoved }
            .sortedBy { it.order }
            .map { it.asset.toSitelink() }
            .toList()
        val newSitelinkIdsAndOrder = sitelinksToSet.mapIndexed { index, sitelink -> sitelinkToId[sitelink]!! to index }

        return (titleIdsToUpdate + textIdsToUpdate + newSitelinkIdsAndOrder).toMap()
    }

    private fun createAbsentNonMediaContents(
        assetContainer: AssetContainer,
        request: PatchCampaignInternalRequest,
        clientId: ClientId,
    ): CreatedAssets {
        val oldTitles = assetContainer.titles.asSequence().map { it.asset.title }.toSet()
        val newTitlesToAdd = (request.titles ?: emptyList()).filter { !oldTitles.contains(it) }
        val createTitlesSubRequests = getCreateTitlesSubRequests(newTitlesToAdd, clientId.asLong())
        val oldTexts = assetContainer.texts.asSequence().map { it.asset.text }.toSet()
        val newTextsToAdd = (request.texts ?: emptyList()).filter { !oldTexts.contains(it) }
        val createTextsSubRequests = getCreateTextsSubRequests(newTextsToAdd, clientId.asLong())
        val oldSitelinks = assetContainer.sitelinks.asSequence().map { it.asset.toSitelink() }.toSet()
        val newSitelinksToAdd = (request.sitelinks ?: emptyList()).filter { !oldSitelinks.contains(it) }.toSet()
        val createSitelinksSubRequests = getCreateSitelinksSubRequests(newSitelinksToAdd, clientId.asLong())

        val allAssets = createTextsSubRequests + createTitlesSubRequests + createSitelinksSubRequests
        if (allAssets.isEmpty()) {
            return CreatedAssets(emptyMap(), emptyMap(), emptyMap())
        }

        val createdAssetIds = grutApiService.assetGrutApi.createObjects(allAssets)

        val newTitleToId = newTitlesToAdd.mapIndexed { index, newTitle ->
            newTitle to createdAssetIds[index].toIdString()
        }.toMap()
        val newTextToId = newTextsToAdd.mapIndexed { index, newText ->
            newText to createdAssetIds[index + newTitlesToAdd.size].toIdString()
        }.toMap()
        val newSitelinkToId = newSitelinksToAdd.mapIndexed { index, newSitelink ->
            newSitelink to createdAssetIds[index + newTitlesToAdd.size + newTextsToAdd.size].toIdString()
        }.toMap()

        return CreatedAssets(newTitleToId = newTitleToId, newTextToId = newTextToId, newSitelinkToId = newSitelinkToId)
    }

    private fun SitelinkAsset.toSitelink() = Sitelink(title, href, description)

    private fun getCreateTitlesSubRequests(titles: Collection<String>, clientId: Long): List<TitleAssetGrut> {
        return titles.map {
            TitleAssetGrut(
                id = generateRandomIdLong(),
                clientId = clientId,
                title = it
            )
        }
    }

    private fun getCreateTextsSubRequests(texts: Collection<String>, clientId: Long): List<TextAssetGrut> {
        return texts.map {
            TextAssetGrut(
                id = generateRandomIdLong(),
                clientId = clientId,
                text = it
            )
        }
    }

    private fun getCreateSitelinksSubRequests(sitelinks: Collection<Sitelink>, clientId: Long): List<SitelinkAssetGrut> {
        return sitelinks.map { sl ->
            SitelinkAssetGrut(
                id = generateRandomIdLong(),
                clientId = clientId,
                title = sl.title,
                href = sl.href,
                description = sl.description
            )
        }
    }

    private fun updateAssetLinks(
        oldActualAssetLinks: Collection<UacYdbCampaignContent>,
        removedAssetIds: Set<String>,
        createdAssetIds: Set<String>,
        newAssetIds: Set<String>,
        updatedUacYdbCampaign: UacYdbCampaign,
        assetIdToOrder: Map<String, Int>,
        newAssetLinkIdsGenerationEnabled: Boolean,
    ): List<UacYdbCampaignContent> {
        val oldAssetIds = createdAssetIds + removedAssetIds
        val assetIdsToCreate = newAssetIds - oldAssetIds
        val assetIdsToRestore = removedAssetIds.intersect(newAssetIds)
        val assetIdsToDelete = createdAssetIds - newAssetIds

        val updatedAssetLinks =
            oldActualAssetLinks + assetIdsToCreate.map { assetId ->
                UacYdbCampaignContent(
                    id = if (newAssetLinkIdsGenerationEnabled) UacYdbUtils.generateUniqueRandomId() else assetId,
                    contentId = assetId,
                    order = 0,
                    status = CampaignContentStatus.CREATED,
                    createdAt = LocalDateTime.now(),
                    removedAt = null,
                    linkType = AssetLinkType.NOT_SPECIFED,
                    // поля ниже в груте не нужны, поэтому не заполняем их
                    campaignId = "",
                    type = null,
                )
            }

        return updatedAssetLinks.mapNotNull { assetLink ->
            getUpdatedAssetLink(
                assetLink, assetIdToOrder,
                assetIdsToCreate = assetIdsToCreate,
                assetIdsToRestore = assetIdsToRestore,
                assetIdsToDelete = assetIdsToDelete,
                updatedUacYdbCampaign
            )
        }
    }

    private fun getUpdatedAssetLink(
        oldAssetLink: UacYdbCampaignContent,
        assetIdToOrder: Map<String, Int>,
        assetIdsToCreate: Set<String>,
        assetIdsToRestore: Set<String>,
        assetIdsToDelete: Set<String>,
        updatedUacYdbCampaign: UacYdbCampaign,
    ): UacYdbCampaignContent? {
        val assetId = oldAssetLink.contentId
        if (assetIdsToDelete.contains(assetId) && updatedUacYdbCampaign.isDraft) {
            return null
        }

        val updatedStatus =
            if (assetIdsToCreate.contains(assetId) || assetIdsToRestore.contains(assetId)) {
                CampaignContentStatus.CREATED
            } else if (assetIdsToDelete.contains(assetId)) {
                CampaignContentStatus.DELETED
            } else {
                oldAssetLink.status
            }
        val updatedRemovedAt =
            if (assetIdsToCreate.contains(assetId) || assetIdsToRestore.contains(assetId)) {
                null
            } else if (assetIdsToDelete.contains(assetId)) {
                LocalDateTime.now()
            } else {
                oldAssetLink.removedAt
            }

        return oldAssetLink.copy(
            order = assetIdToOrder[assetId] ?: 0,
            status = updatedStatus,
            removedAt = updatedRemovedAt,
        )
    }

    private fun shouldNotRestoreOldLinks(
        updatedUacYdbCampaign: UacYdbCampaign,
        newAssetLinkIdsGenerationEnabled: Boolean,
    ): Boolean {
        if (!newAssetLinkIdsGenerationEnabled) {
            return false
        }
        if (updatedUacYdbCampaign.advType != AdvType.TEXT) {
            return false
        }
        return true
    }
}
