package ru.yandex.direct.web.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.http.ResponseEntity
import org.springframework.stereotype.Service
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.CreativeType
import ru.yandex.direct.core.entity.uac.model.UacStrategyPlatform.SEARCH
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.uac.service.GrutUacContentService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.result.Result
import ru.yandex.direct.web.entity.uac.contentsEmptyResponse
import ru.yandex.direct.web.entity.uac.model.CreateContentRequest
import ru.yandex.direct.web.entity.uac.textsEmptyResponse
import ru.yandex.direct.web.entity.uac.titlesEmptyResponse
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService
import java.util.Locale

@Service
class GrutUacContentWebService(
    private val grutUacContentService: GrutUacContentService,
    validationResultConversionService: ValidationResultConversionService,
    dbQueueRepository: DbQueueRepository,
    shardHelper: ShardHelper,
    uacImageContentUploader: UacImageContentUploader,
    uacVideoContentUploader: UacVideoContentUploader,
    uacHtml5ContentUploader: UacHtml5ContentUploader,
) : BaseUacContentWebService(uacImageContentUploader, uacVideoContentUploader, uacHtml5ContentUploader, validationResultConversionService, dbQueueRepository, shardHelper) {

    companion object {
        private val logger = LoggerFactory.getLogger(GrutUacContentWebService::class.java)
    }

    override fun checkCampaignIsCompleteForUpdateStatus(campaign: UacYdbCampaign): ResponseEntity<Any>? {
        val assetContainer = grutUacContentService.getAssetContainer(campaign)
        if (campaign.advType == AdvType.MOBILE_CONTENT) {
            if (assetContainer.mediaAssets.isEmpty() && campaign.strategyPlatform != SEARCH) {
                return contentsEmptyResponse()
            }
            val isOnlyPlayableContents = assetContainer.mediaAssets.size == assetContainer.html5s.size
            if (!isOnlyPlayableContents && assetContainer.texts.isEmpty()) {
                return textsEmptyResponse()
            }
            if (!isOnlyPlayableContents && assetContainer.titles.isEmpty()) {
                return titlesEmptyResponse()
            }
        }

        if (campaign.advType == AdvType.TEXT) {
            if (assetContainer.texts.isEmpty()) {
                return textsEmptyResponse()
            }
            if (assetContainer.titles.isEmpty()) {
                return titlesEmptyResponse()
            }
        }

        return checkCampaignIsCompleteCommon(campaign)
    }

    override fun fillContent(ydbContent: UacYdbContent?): Content? {
        return grutUacContentService.fillContent(ydbContent)
    }

    override fun insertContent(content: UacYdbContent) {
        grutUacContentService.insertContent(content)
    }

    override fun uploadContent(
        operator: User,
        subjectUser: User,
        creativeType: CreativeType,
        multipartFile: MultipartFile?,
        createContentRequest: CreateContentRequest?,
        locale: Locale?,
    ): Result<UacYdbContent> {
        val accountId = subjectUser.clientId.toString()

        return if (createContentRequest != null) {
            uploadContentFromRequest(subjectUser, accountId, createContentRequest, creativeType, locale)
        } else {
            uploadContentFromFile(subjectUser, accountId, multipartFile!!, creativeType, locale, null)
        }
    }
}
