package ru.yandex.direct.web.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargetingAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifier
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.uac.converter.UacBidModifiersConverter.toBidModifiers
import ru.yandex.direct.core.entity.uac.model.UacAdjustmentRequest
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.feature.FeatureName

@Service
class UacAdjustmentsService(
    private val bidModifierService: BidModifierService,
    private val featureService: FeatureService,
) {
    companion object {
        val AVAILABLE_TYPES = setOf(
            BidModifierType.DEMOGRAPHY_MULTIPLIER,
            BidModifierType.GEO_MULTIPLIER,
            BidModifierType.RETARGETING_MULTIPLIER,
            BidModifierType.RETARGETING_FILTER,
        )
    }

    fun updateBidModifiers(
        directCampaignId: Long,
        operator: User,
        subjectUser: User,
        bidModifiers: List<UacAdjustmentRequest>?,
        retargetingConditionId: Long?,
    ) {
        val existingBidModifiers = bidModifierService.getByCampaignIds(
            subjectUser.clientId,
            listOf(directCampaignId),
            AVAILABLE_TYPES,
            setOf(BidModifierLevel.CAMPAIGN),
            operator.uid
        )

        val isUpdated = isRegionalModifierUpdated(existingBidModifiers, bidModifiers, directCampaignId)
            || isDemographicsModifierUpdated(existingBidModifiers, bidModifiers, directCampaignId)
            || isRetargetingModifierUpdated(existingBidModifiers, bidModifiers, directCampaignId)
            || isRetargetingFilterUpdated(existingBidModifiers, retargetingConditionId)

        if (isUpdated) {
            bidModifierService.replaceModifiers(
                subjectUser.clientId,
                operator.uid,
                toBidModifiers(
                    bidModifiers,
                    retargetingConditionId,
                    directCampaignId,
                    featureService.isEnabledForClientId(subjectUser.clientId, FeatureName.SEARCH_RETARGETING_ENABLED)
                ),
                setOf(CampaignIdAndAdGroupIdPair().withCampaignId(directCampaignId).withAdGroupId(null))
            )
        }
    }

    private fun isRegionalModifierUpdated(
        existingBidModifiers: List<BidModifier>,
        bidModifiers: List<UacAdjustmentRequest>?,
        directCampaignId: Long,
    ): Boolean {
        val toBidModifierByRegion = { bidModifierRegionalAdjustment: List<BidModifierRegionalAdjustment> ->
            bidModifierRegionalAdjustment
                .filter { it.hidden == false }
                .associate { it.regionId to (it as BidModifierAdjustment) }
        }
        val toBidModifiersRegionalAdjustment = { modifiers: List<BidModifier> ->
            modifiers
                .filterIsInstance<BidModifierGeo>()
                .map { toBidModifierByRegion.invoke(it.regionalAdjustments) }
                .flatMap { it.entries }
                .associate { it.key to it.value }
                .toMutableMap()
        }

        val existingBidModifiersRegionalAdjustment = toBidModifiersRegionalAdjustment(existingBidModifiers)
        val bidModifiersRegionalAdjustments =
            toBidModifiersRegionalAdjustment(
                toBidModifiers(bidModifiers, null, directCampaignId, false)
            )
        return isModifierUpdated(existingBidModifiersRegionalAdjustment, bidModifiersRegionalAdjustments)
    }

    private fun isDemographicsModifierUpdated(
        existingBidModifiers: List<BidModifier>,
        bidModifiers: List<UacAdjustmentRequest>?,
        directCampaignId: Long,
    ): Boolean {
        val toBidModifierByDemographic = { bidModifierDemographicsAdjustment: List<BidModifierDemographicsAdjustment> ->
            bidModifierDemographicsAdjustment
                .associate { (it.age to it.gender) to (it as BidModifierAdjustment) }
        }
        val toBidModifiersDemographicsAdjustment = { modifiers: List<BidModifier> ->
            modifiers
                .filterIsInstance<BidModifierDemographics>()
                .map { toBidModifierByDemographic.invoke(it.demographicsAdjustments) }
                .flatMap { it.entries }
                .associate { it.key to it.value }
                .toMutableMap()
        }
        val existingBidModifiersDemographicsAdjustments = toBidModifiersDemographicsAdjustment(existingBidModifiers)
        val bidModifierDemographicAdjustments =
            toBidModifiersDemographicsAdjustment(
                toBidModifiers(bidModifiers, null, directCampaignId, false)
            )
        return isModifierUpdated(existingBidModifiersDemographicsAdjustments, bidModifierDemographicAdjustments)
    }

    private fun isRetargetingModifierUpdated(
        existingBidModifiers: List<BidModifier>,
        bidModifiers: List<UacAdjustmentRequest>?,
        directCampaignId: Long,
    ): Boolean {
        val toBidModifierByRetargeting =
            { bidModifierRetargetingAdjustment: List<AbstractBidModifierRetargetingAdjustment> ->
                bidModifierRetargetingAdjustment
                    .associate { it.retargetingConditionId to (it as BidModifierAdjustment) }
            }
        val toBidModifiersRetargetingAdjustment = { modifiers: List<BidModifier> ->
            modifiers
                .filterIsInstance<BidModifierRetargeting>()
                .map { toBidModifierByRetargeting.invoke(it.retargetingAdjustments) }
                .flatMap { it.entries }
                .associate { it.key to it.value }
                .toMutableMap()
        }
        val existingBidModifiersDemographicsAdjustments = toBidModifiersRetargetingAdjustment(existingBidModifiers)
        val bidModifierDemographicAdjustments =
            toBidModifiersRetargetingAdjustment(
                toBidModifiers(bidModifiers, null, directCampaignId, false)
            )
        return isModifierUpdated(existingBidModifiersDemographicsAdjustments, bidModifierDemographicAdjustments)
    }

    private fun isRetargetingFilterUpdated(
        existingBidModifiers: List<BidModifier>,
        retargetingConditionId: Long?,
    ): Boolean  = existingBidModifiers
        .filterIsInstance<BidModifierRetargetingFilter>()
        .firstOrNull()
        ?.retargetingAdjustments
        ?.any { it.retargetingConditionId != retargetingConditionId } ?: (retargetingConditionId != null)

    private fun isModifierUpdated(
        existingBidModifierAdjustments: MutableMap<*, BidModifierAdjustment>,
        bidModifierAdjustments: MutableMap<*, BidModifierAdjustment>,
    ) = existingBidModifierAdjustments.size != bidModifierAdjustments.size
        || !existingBidModifierAdjustments.keys.containsAll(bidModifierAdjustments.keys)
        || bidModifierAdjustments.keys
        .any {
            bidModifierAdjustments[it]!!.percent != existingBidModifierAdjustments[it]!!.percent
        }

    fun isValidAdjustment(adjustment: UacAdjustmentRequest): Boolean {
        return isGeoAdjustment(adjustment)
            || isDemographicsAdjustment(adjustment)
            || isRetargetingAdjustment(adjustment)
    }

    private fun isGeoAdjustment(adjustment: UacAdjustmentRequest): Boolean {
        return adjustment.region != null
            && adjustment.age == null
            && adjustment.gender == null
            && adjustment.retargetingConditionId == null
    }

    private fun isDemographicsAdjustment(adjustment: UacAdjustmentRequest): Boolean {
        return adjustment.region == null
            && (adjustment.age != null || adjustment.gender != null)
            && adjustment.retargetingConditionId == null
    }

    private fun isRetargetingAdjustment(adjustment: UacAdjustmentRequest): Boolean {
        return adjustment.region == null
            && adjustment.age == null
            && adjustment.gender == null
            && adjustment.retargetingConditionId != null
    }
}
