package ru.yandex.direct.web.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.appmetrika.AppMetrikaClient
import ru.yandex.direct.appmetrika.model.AppMetrikaClientException
import ru.yandex.direct.appmetrika.model.response.UniversalCampaignInfo
import ru.yandex.direct.appmetrika.model.request.UniversalCampaignRequest
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.direct.web.entity.uac.converter.UacAppMetrikaAppsConverter
import ru.yandex.direct.web.entity.uac.model.UacAppMetrikaApp

@Service
class UacAppMetrikaService(
    private val appMetrikaClient: AppMetrikaClient
) {
    companion object {
        private const val APPLICATIONS_LIMIT = 100
        private val logger = LoggerFactory.getLogger(UacAppMetrikaService::class.java)
    }

    fun getApps(user: User): List<UacAppMetrikaApp> {
        val applications = appMetrikaClient.getApplications(
            user.uid, null, null, null, APPLICATIONS_LIMIT, null
        )
        return applications
            .flatMap { it.bundleIds }
            .map { UacAppMetrikaAppsConverter.toUacAppMetrikaApp(it) }
    }

    fun createAppMetrikaUrl(appInfo: AppInfo, user: User): UniversalCampaignInfo? {
        val request = UniversalCampaignRequest().apply {
            campaignName = null
            platforms = UniversalCampaignRequestPlatforms().apply {
                android = if (appInfo.platform == Platform.ANDROID) UniversalCampaignRequestPlatformInfo().apply {
                    bundleId = appInfo.bundleId
                } else null
                ios = if (appInfo.platform == Platform.IOS) UniversalCampaignRequestPlatformInfo().apply {
                    bundleId = appInfo.bundleId
                } else null
            }
        }
        try {
            val application = appMetrikaClient.getApplications(
                user.uid, appInfo.bundleId, toAppMetrikaPlatform(appInfo.platform), null, 1, null
            ).firstOrNull() ?: return null
            return appMetrikaClient.getUniversalCampaign(user.uid, application.id, request)
        } catch (e: AppMetrikaClientException) {
            logger.warn("Got an exception when querying for metrika applications for clientId: " + user.clientId, e)
            return null
        } catch (e: InterruptedRuntimeException) {
            logger.warn("Got an exception when querying for metrika applications for clientId: " + user.clientId, e)
            return null
        }
    }

    private fun toAppMetrikaPlatform(uacPlatform: Platform): ru.yandex.direct.appmetrika.model.Platform =
        when (uacPlatform) {
            Platform.ANDROID -> ru.yandex.direct.appmetrika.model.Platform.android
            Platform.IOS -> ru.yandex.direct.appmetrika.model.Platform.ios
        }
}
