package ru.yandex.direct.web.entity.uac.service

import java.time.Duration
import java.time.LocalDateTime
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.CampaignContentStatus
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.Sitelink
import ru.yandex.direct.core.entity.uac.model.Status
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignContentRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.direct.core.entity.uac.service.UacContentService
import ru.yandex.direct.web.entity.uac.model.PatchCampaignInternalRequest

// TODO: переписать без копипасты
@Service
class UacCampaignContentUpdateService(
    private val uacYdbCampaignContentRepository: UacYdbCampaignContentRepository,
    private val uacContentService: UacContentService,
    ppcPropertiesSupport: PpcPropertiesSupport,
) {

    private val doNotRestoreRemovedAssetsInUac: PpcProperty<Boolean> = ppcPropertiesSupport
        .get(PpcPropertyNames.DO_NOT_RESTORE_REMOVED_ASSETS_IN_UAC_TGO, Duration.ofMinutes(5))

    fun updateCampaignContents(
        uacCampaign: UacYdbCampaign,
        request: PatchCampaignInternalRequest,
        multipleAdsInUc: Boolean,
    ) {
        val doRestoreAssets = !multipleAdsInUc
            || uacCampaign.advType != AdvType.TEXT
            || !doNotRestoreRemovedAssetsInUac.getOrDefault(false)

        updateMediaCampaignContents(uacCampaign, request.contentIds, doRestoreAssets)
        updateTextsCampaignContents(uacCampaign, request.texts, MediaType.TEXT, doRestoreAssets)
        updateTextsCampaignContents(uacCampaign, request.titles, MediaType.TITLE, doRestoreAssets)
        updateSitelinksCampaignContents(uacCampaign, request.sitelinks, doRestoreAssets)
    }

    private fun updateSitelinksCampaignContents(
        uacCampaign: UacYdbCampaign,
        sitelinks: List<Sitelink>?,
        doRestoreAssets: Boolean,
    ) {
        if (sitelinks == null) {
            return
        }

        val oldActiveSitelinks = mutableListOf<Sitelink>()
        val oldRemovedSitelinks = mutableSetOf<Sitelink>()
        val newActiveSitelinks = sitelinks.toSet()

        uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
            .filter { it.type == MediaType.SITELINK }
            .sortedBy { it.order }
            .forEach {
                if (it.removedAt == null) {
                    oldActiveSitelinks.add(it.sitelink!!)
                } else if (doRestoreAssets) {
                    oldRemovedSitelinks.add(it.sitelink!!)
                }
            }

        if (newActiveSitelinks == oldActiveSitelinks) {
            return
        }

        val oldSitelinks = oldActiveSitelinks.union(oldRemovedSitelinks)
        val sitelinksForAdd = newActiveSitelinks.subtract(oldSitelinks)
        val sitelinksForUpdate = newActiveSitelinks.intersect(oldRemovedSitelinks)
        val sitelinksForDelete = oldActiveSitelinks.subtract(newActiveSitelinks)

        if (sitelinksForUpdate.isNotEmpty()) {
            updateCampaignContents(uacCampaign = uacCampaign, sitelinks = sitelinksForUpdate, newRemovedAt = null)
        }
        if (sitelinksForAdd.isNotEmpty()) {
            createCampaignContents(uacCampaign = uacCampaign, sitelinks = sitelinksForAdd)
        }
        if (sitelinksForDelete.isNotEmpty()) {
            if (uacCampaign.isDraft) {
                deleteCampaignContents(uacCampaign = uacCampaign, sitelinks = sitelinksForDelete)
            } else {
                updateCampaignContents(uacCampaign = uacCampaign, sitelinks = sitelinksForDelete, newRemovedAt = LocalDateTime.now())
            }
        }

        val campaignContents = uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
        sitelinks.forEachIndexed { index, sitelink ->
            val campaignContentToUpdate = campaignContents.find { campaignContent ->
                campaignContent.type == MediaType.SITELINK && campaignContent.sitelink == sitelink
            }!!
            uacYdbCampaignContentRepository.updateOrderForCampaignContent(campaignContentToUpdate.id, index)
        }
    }

    private fun updateTextsCampaignContents(
        uacCampaign: UacYdbCampaign,
        texts: List<String>?,
        mediaType: MediaType,
        doRestoreAssets: Boolean,
    ) {
        if (texts == null) {
            return
        }

        val oldActiveTexts = mutableListOf<String>()
        val oldRemovedTexts = mutableSetOf<String>()
        val newActiveTexts = texts.toSet()

        uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
            .filter { it.type == mediaType }
            .sortedBy { it.order }
            .forEach {
                if (it.removedAt == null) {
                    oldActiveTexts.add(it.text!!)
                } else if (doRestoreAssets) {
                    oldRemovedTexts.add(it.text!!)
                }
            }

        if (texts == oldActiveTexts) {
            return
        }

        val oldTexts = oldActiveTexts.union(oldRemovedTexts)
        val textsForAdd = newActiveTexts.subtract(oldTexts)
        val textsForUpdate = newActiveTexts.intersect(oldRemovedTexts)
        val textsForDelete = oldActiveTexts.subtract(newActiveTexts)

        if (textsForUpdate.isNotEmpty()) {
            updateCampaignContents(uacCampaign = uacCampaign, texts = textsForUpdate, mediaType = mediaType, newRemovedAt = null)
        }
        if (textsForAdd.isNotEmpty()) {
            createCampaignContents(uacCampaign, textsForAdd, mediaType)
        }
        if (textsForDelete.isNotEmpty()) {
            if (uacCampaign.isDraft) {
                deleteCampaignContents(uacCampaign = uacCampaign, texts = textsForDelete, mediaType = mediaType)
            } else {
                updateCampaignContents(uacCampaign = uacCampaign, texts = textsForDelete, mediaType = mediaType, newRemovedAt = LocalDateTime.now())
            }
        }

        val campaignContents = uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
        texts.forEachIndexed { index, text ->
            val campaignContentToUpdate = campaignContents.find { campaignContent ->
                campaignContent.text == text && campaignContent.type == mediaType
            }!!
            uacYdbCampaignContentRepository.updateOrderForCampaignContent(campaignContentToUpdate.id, index)
        }
    }

    private fun deleteCampaignContents(
        uacCampaign: UacYdbCampaign,
        texts: Set<String>? = null,
        mediaType: MediaType? = null,
        contentIds: Set<String>? = null,
        sitelinks: Set<Sitelink>? = null,
    ) {
        val campaignContentIdsToDelete = if (texts != null) {
            uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
                .filter { it.type == mediaType && it.text in texts }
                .map { it.id }
        } else if (contentIds != null) {
            uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
                .filter { it.contentId in contentIds }
                .map { it.id }
        } else if (sitelinks != null) {
            uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
                .filter { it.type == MediaType.SITELINK && it.sitelink in sitelinks }
                .map { it.id }
        } else {
            emptyList()
        }

        uacYdbCampaignContentRepository.delete(campaignContentIdsToDelete)
    }

    private fun createCampaignContents(
        uacCampaign: UacYdbCampaign,
        texts: Set<String>? = null,
        mediaType: MediaType? = null,
        contents: Collection<Content>? = null,
        sitelinks: Collection<Sitelink>? = null,
    ) {
        val uacCampaignContentsToCreate = if (texts != null) {
            texts.mapIndexed { index, text ->
                UacYdbCampaignContent(
                    campaignId = uacCampaign.id,
                    order = index,
                    type = mediaType!!,
                    status = CampaignContentStatus.CREATED,
                    text = text,
                )
            }
        } else if (contents != null) {
            contents.mapIndexed { index, content ->
                UacYdbCampaignContent(
                    campaignId = uacCampaign.id,
                    order = index,
                    type = content.type,
                    status = CampaignContentStatus.CREATED,
                    contentId = content.id,
                )
            }
        } else if (sitelinks != null) {
            sitelinks.mapIndexed { index, sitelink ->
                UacYdbCampaignContent(
                    campaignId = uacCampaign.id,
                    order = index,
                    type = MediaType.SITELINK,
                    status = CampaignContentStatus.CREATED,
                    sitelink = sitelink,
                )
            }
        } else {
            emptyList()
        }

        uacYdbCampaignContentRepository.addCampaignContents(uacCampaignContentsToCreate)
    }

    private fun updateCampaignContents(
        uacCampaign: UacYdbCampaign,
        texts: Set<String>? = null,
        mediaType: MediaType? = null,
        contentIds: Set<String>? = null,
        sitelinks: Set<Sitelink>? = null,
        newRemovedAt: LocalDateTime?,
    ) {
        val newStatus = if (newRemovedAt == null) {
            CampaignContentStatus.CREATED
        } else {
            CampaignContentStatus.DELETED
        }

        val campaignContentIdsToUpdate = if (texts != null) {
            uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
                .filter { it.type == mediaType && it.text in texts }
                .map { it.id }
        } else if (contentIds != null) {
            uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
                .filter { it.contentId in contentIds }
                .map { it.id }
        } else if (sitelinks != null) {
            uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
                .filter { it.sitelink in sitelinks }
                .map { it.id }
        } else {
            emptyList()
        }

        uacYdbCampaignContentRepository.update(campaignContentIdsToUpdate, newRemovedAt, newStatus)
    }

    private fun updateMediaCampaignContents(
        uacCampaign: UacYdbCampaign,
        contentIds: List<String>?,
        doRestoreAssets: Boolean,
    ) {
        if (contentIds == null) {
            return
        }

        val oldActiveContentIds = mutableListOf<String>()
        val oldRemovedContentIds = mutableSetOf<String>()
        val newActiveContentIds = contentIds.toSet()

        uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
            .filter { it.contentId != null }
            .sortedBy { it.order }
            .forEach {
                if (it.removedAt == null) {
                    oldActiveContentIds.add(it.contentId!!)
                } else if (doRestoreAssets) {
                    oldRemovedContentIds.add(it.contentId!!)
                }
            }

        if (newActiveContentIds == oldActiveContentIds) {
            return
        }

        val oldContentIds = oldActiveContentIds.union(oldRemovedContentIds)
        val contentsForCreate = uacContentService.getContents(newActiveContentIds.subtract(oldContentIds))
        val contentsForUpdate = uacContentService.getContents(oldRemovedContentIds.intersect(newActiveContentIds))
        val contentsForDelete = uacContentService.getContents(oldActiveContentIds.subtract(newActiveContentIds))
        val contentIdsForUpdate = contentsForUpdate.map { it.id }.toSet()
        val contentIdsForDelete = contentsForDelete.map { it.id }.toSet()

        if (contentsForCreate.isNotEmpty()) {
            createCampaignContents(uacCampaign = uacCampaign, contents = contentsForCreate)
        }
        if (contentsForUpdate.isNotEmpty()) {
            updateCampaignContents(uacCampaign = uacCampaign, contentIds = contentIdsForUpdate, newRemovedAt = null)
        }
        if (contentsForDelete.isNotEmpty()) {
            if (uacCampaign.isDraft) {
                deleteCampaignContents(uacCampaign = uacCampaign, contentIds = contentIdsForDelete)
            } else {
                updateCampaignContents(uacCampaign = uacCampaign, contentIds = contentIdsForDelete, newRemovedAt = LocalDateTime.now())
            }
        }

        val campaignContentsById = uacYdbCampaignContentRepository.getCampaignContents(uacCampaign.id)
            .associateBy { it.contentId }
        contentIds.forEachIndexed { index, contentId ->
            val campaignContentToUpdate = campaignContentsById[contentId]!!
            uacYdbCampaignContentRepository.updateOrderForCampaignContent(campaignContentToUpdate.id, index)
        }
    }
}
