package ru.yandex.direct.web.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup
import ru.yandex.direct.core.entity.adgroup.container.ComplexMobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.GeoproductAvailability
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.service.complex.cpm.AddComplexCpmAdGroupValidationService
import ru.yandex.direct.core.entity.banner.model.ButtonAction
import ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstants.MAX_NUMBER_OF_NARROW_CHARACTERS
import ru.yandex.direct.core.entity.banner.service.validation.BannerTextValidator
import ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants.MAX_LENGTH_BODY
import ru.yandex.direct.core.entity.banner.type.body.BannerWithBodyConstants.MAX_LENGTH_BODY_WORD
import ru.yandex.direct.core.entity.banner.type.button.BannerWithButtonHelper
import ru.yandex.direct.core.entity.banner.type.pixels.BannerPixelsValidator
import ru.yandex.direct.core.entity.banner.type.pixels.ClientPixelProviderRepository
import ru.yandex.direct.core.entity.banner.type.pixels.NewPixelsValidationContext
import ru.yandex.direct.core.entity.banner.type.pixels.PixelPermissionInfo
import ru.yandex.direct.core.entity.banner.type.title.BannerConstantsService.MAX_LENGTH_TITLE_WORD
import ru.yandex.direct.core.entity.banner.type.title.BannerConstantsService.NEW_MAX_LENGTH_TITLE
import ru.yandex.direct.core.entity.banner.type.titleextension.BannerWithTitleExtensionValidatorProvider
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSettings
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.client.service.ClientMeasurerSettingsService
import ru.yandex.direct.core.entity.client.service.MediascopeClientSettingsService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.keyword.model.Keyword
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.retargeting.service.validation2.AddRetargetingConditionValidationService2
import ru.yandex.direct.core.entity.uac.model.AdvType
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.grid.processing.service.campaign.uc.UcCampaignValidationService
import ru.yandex.direct.result.Result
import ru.yandex.direct.utils.CommonUtils.nvl
import ru.yandex.direct.utils.FunctionalUtils.mapList
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.PathHelper
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder
import ru.yandex.direct.web.entity.uac.converter.UacGroupsConverter
import ru.yandex.direct.web.entity.uac.model.UacModifyCampaignDataContainer
import ru.yandex.direct.web.entity.uac.validation.CpmUacMeasurersValidator

@Service
class UacCampaignValidationService(
    private val addComplexCpmAdGroupValidationService: AddComplexCpmAdGroupValidationService,
    private val retargetingConditionValidationService: AddRetargetingConditionValidationService2,
    private val ucCampaignValidationService: UcCampaignValidationService,
    private val bannerWithTitleExtensionValidatorProvider: BannerWithTitleExtensionValidatorProvider,
    private val clientGeoService: ClientGeoService,
    private val clientMeasurerSettingsService: ClientMeasurerSettingsService,
    private val mediascopeClientSettingsService: MediascopeClientSettingsService,
    private val featureService: FeatureService,
    private val shardHelper: ShardHelper,
    private val clientPixelProviderRepository: ClientPixelProviderRepository,
    private val bannerWithButtonHelper: BannerWithButtonHelper,

    ) {

    fun getTitleValidator(): BannerTextValidator {
        return BannerTextValidator.builder(NEW_MAX_LENGTH_TITLE, MAX_LENGTH_TITLE_WORD).build()
    }

    fun getTitleExtensionValidator(): Validator<String, Defect<*>> {
        return bannerWithTitleExtensionValidatorProvider.titleExtensionValidator()
    }

    fun getTextValidator(advType: AdvType): BannerTextValidator {
        val validatorBuilder = BannerTextValidator.builder(MAX_LENGTH_BODY, MAX_LENGTH_BODY_WORD)

        if (advType != AdvType.MOBILE_CONTENT) {
            validatorBuilder.withMaxNumberOfNarrowCharacters(MAX_NUMBER_OF_NARROW_CHARACTERS)
        }

        return validatorBuilder.build()
    }

    fun getMinusKeywordsValidator(): MinusPhraseValidator {
        return MinusPhraseValidator.minusKeywordIsValid(MinusPhraseValidator.ValidationMode.ONE_ERROR_PER_TYPE_AND_KEYWORD)
    }

    fun validateAddingCpmGroup(
        createDataContainer: UacModifyCampaignDataContainer,
        contents: Collection<Content>,
        clientId: ClientId,
    ): ValidationResult<AdGroup, Defect<*>> {
        val complexGroup = UacGroupsConverter.toComplexCpmAdgroup(
            createDataContainer, contents, clientId.asLong()
        )

        val adGroupVr = addComplexCpmAdGroupValidationService.validateAdGroup(
            complexGroup, mapOf(complexGroup.adGroup.id to GeoproductAvailability.EMPTY), emptyMap()
        )

        if (createDataContainer.retargetingCondition != null) {
            val retargetingConditionVr = retargetingConditionValidationService.validate(
                complexGroup.retargetingConditions.map { it as RetargetingCondition },
                clientId
            )

            adGroupVr.addSubResult(PathHelper.field(ComplexCpmAdGroup.RETARGETING_CONDITIONS), retargetingConditionVr)
        }

        createDataContainer.cpmAssets?.entries?.forEach {
            val cpmPixelsVr = getPixelsValidator(clientId, complexGroup).apply(it.value.pixels)
            adGroupVr.addSubResult(PathHelper.field(String.format("cpmAssets.%s.pixels", it.key)), cpmPixelsVr)
        }

        return adGroupVr
    }

    fun validateUacAdGroup(
        clientId: ClientId,
        keywords: List<String>?,
        existingKeywords: List<String>?,
        geo: List<Long>,
    ): Result<Any> {
        val name = String.format("Ad group %d", clientId.asLong())
        val clientGeoTree = clientGeoService.getClientTranslocalGeoTree(clientId)
        val adGroup = MobileContentAdGroup().withType(AdGroupType.MOBILE_CONTENT).withName(name).withGeo(geo)
        val keywordsVr = ucCampaignValidationService.preValidateKeywords(
            clientId,
            adGroup,
            keywords?.map { Keyword().withPhrase(it) } ?: listOf(),
            existingKeywords?.map { Keyword().withPhrase(it) } ?: listOf())
        val complexAdGroupVr = ModelItemValidationBuilder.of(ComplexMobileContentAdGroup().withAdGroup(adGroup))
            .result
            .addSubResult(PathHelper.field(ComplexMobileContentAdGroup.KEYWORDS), keywordsVr)
        if (geo.isNotEmpty()) {
            val adGroupVr =
                ucCampaignValidationService.preValidateAddingAdGroup(adGroup, clientId, clientGeoTree, false)
            complexAdGroupVr.addSubResult(PathHelper.field(ComplexMobileContentAdGroup.AD_GROUP), adGroupVr)
        }
        if (complexAdGroupVr.hasAnyErrors()) {
            return Result.broken(complexAdGroupVr)
        }
        return Result.successful(null)
    }

    fun getCpmUacMeasurersValidator(clientId: ClientId): CpmUacMeasurersValidator {
        val clientSettings: ClientMeasurerSettings? =
            clientMeasurerSettingsService.getByClientIdAndSystem(clientId.asLong(), ClientMeasurerSystem.MEDIASCOPE)
        return CpmUacMeasurersValidator(clientSettings != null &&
            !mediascopeClientSettingsService.isExpired(clientSettings.settings))
    }

    fun getPermissionInfosForClient(shard: Int, clientId: ClientId): List<PixelPermissionInfo> {
        return mapList(clientPixelProviderRepository.getClientPixelProviders(shard, clientId.asLong()),
            PixelPermissionInfo::fromDb)
    }

    fun getPixelsValidator(clientId: ClientId, complexCpmAdGroup: ComplexCpmAdGroup): BannerPixelsValidator {
        val shard: Int = shardHelper.getShardByClientId(clientId)
        val context = NewPixelsValidationContext(
            getPermissionInfosForClient(shard, clientId),
            CampaignType.CPM_BANNER,
            emptyList(), //список Placement используется только в кампаниях-сделках, их как минимум нет в uac, как максимум планировали оторвать
            nvl(complexCpmAdGroup.retargetingConditions, emptyList()).map { it as RetargetingCondition },
            !nvl(complexCpmAdGroup.keywords, emptyList()).isEmpty(),
            if (complexCpmAdGroup.adGroup != null) complexCpmAdGroup.adGroup.type else AdGroupType.CPM_BANNER)
        return BannerPixelsValidator.bannerPixelsValidator(context)
    }

    fun getAllowedButtonActions(clientId: ClientId): Set<ButtonAction> {
        return bannerWithButtonHelper.getAllowedButtonActions(featureService.getEnabledForClientId(clientId))
    }
}
