package ru.yandex.direct.web.entity.uac.service

import java.time.LocalDate
import org.springframework.stereotype.Service
import ru.yandex.direct.common.util.HttpUtil
import ru.yandex.direct.core.entity.uac.UacCommonUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.intapi.client.IntApiClient
import ru.yandex.direct.intapi.client.model.request.statistics.CampaignStatisticsRequest
import ru.yandex.direct.intapi.client.model.request.statistics.ReportOptions
import ru.yandex.direct.intapi.client.model.request.statistics.ReportType
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionColumn
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupBy
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate
import ru.yandex.direct.intapi.client.model.response.statistics.CampaignStatisticsItem
import ru.yandex.direct.intapi.client.model.response.statistics.CampaignStatisticsResponse
import ru.yandex.direct.web.entity.uac.model.CreativeStatistics

@Service
class UacCpmStatisticsService(
    private val intApiClient: IntApiClient

) {
    private fun makeStatisticsRequestForCpm(
        clientUid: Long,
        operatorUid: Long,
        directCampaignId: Long,
        dateFrom: LocalDate,
        dateTo: LocalDate
    ): CampaignStatisticsResponse {
        val request: CampaignStatisticsRequest = CampaignStatisticsRequest()
            .withUid(clientUid)
            .withOperatorUid(operatorUid)
            .withReportOptions(
                ReportOptions()
                    .withCampaignId(directCampaignId)
                    .withFilters(emptyMap())
                    .withColumns(
                        setOf(
                            ReportOptionColumn.CLICKS,
                            ReportOptionColumn.SHOWS,
                            ReportOptionColumn.COST,
                            ReportOptionColumn.CTR,
                            ReportOptionColumn.VIDEO_AVG_TRUE_VIEW_COST,
                            ReportOptionColumn.AVG_CPM,
                            ReportOptionColumn.AVG_CPC,
                            ReportOptionColumn.UNIQ_VIEWERS,
                            ReportOptionColumn.VIDEO_TRUE_VIEW,
                            ReportOptionColumn.VIDEO_FIRST_QUARTILE_RATE,
                            ReportOptionColumn.VIDEO_MIDPOINT_RATE,
                            ReportOptionColumn.VIDEO_THIRD_QUARTILE_RATE,
                            ReportOptionColumn.VIDEO_COMPLETE_RATE,
                            ReportOptionColumn.AVG_NSHOW,
                            ReportOptionColumn.AVG_NSHOW_COMPLETE,
                        )
                    )
                    .withGroupBy(setOf(ReportOptionGroupBy.CREATIVE_ID))
                    .withGroupByDate(ReportOptionGroupByDate.NONE)
                    .withDateFrom(dateFrom)
                    .withDateTo(dateTo)
                    .withReportType(ReportType.MOC)
            )
            .withLocale(HttpUtil.getCurrentLocale().map { it.language }.orElse(null))

        return intApiClient.getCampaignStatistics(request)
    }

    fun getCreativesStatisticsFromMoc(
        contentsByCampaignContentId: Map<String, UacYdbContent>,
        clientUid: Long,
        operatorUid: Long,
        directCampaignId: Long,
        dateFrom: LocalDate,
        dateTo: LocalDate
    ): Map<String, CreativeStatistics> {
        val statistics = makeStatisticsRequestForCpm(clientUid, operatorUid, directCampaignId, dateFrom, dateTo)

        val contentsByCreativeId: Map<Long, UacYdbContent> =
            contentsByCampaignContentId.values.toSet().associateBy { (it.meta[UacCommonUtils.CREATIVE_ID_KEY] as Number).toLong() }

        val statisticsByContentId = statistics.data
            .filter { contentsByCreativeId[it.creativeId] != null }
            .associateBy { contentsByCreativeId[it.creativeId]!!.id }
            .mapValues { aggregateCreativeStat(it.value) }

        return contentsByCampaignContentId
            .filterValues { statisticsByContentId.containsKey(it.id) }
            .mapValues { statisticsByContentId[it.value.id]!! }
    }

    private fun aggregateCreativeStat(bannersStat: CampaignStatisticsItem): CreativeStatistics {
        return CreativeStatistics(
            clicks = bannersStat.clicks ?: 0,
            shows = bannersStat.shows ?: 0,
            cost = bannersStat.cost ?: 0.0,
            costMicros = 0,
            costCurMicros = 0,
            costTaxFreeMicros = 0,
            ctr = bannersStat.ctr ?: 0.0,
            videoAvgTrueViewCost = bannersStat.videoAvgTrueViewCost ?: 0.0,
            videoTrueView = bannersStat.videoTrueView ?: 0,
            uniqViewers = bannersStat.uniqViewers ?: 0,
            avgCpm = bannersStat.avgCpm ?: 0.0,
            avgCpc = bannersStat.avgCpc ?: 0.0,
            videoFirstQuartileRate = bannersStat.videoFirstQuartileRate ?: 0.0,
            videoMidpointRate = bannersStat.videoMidpointRate ?: 0.0,
            videoThirdQuartileRate = bannersStat.videoThirdQuartileRate ?: 0.0,
            videoCompleteRate = bannersStat.videoCompleteRate ?: 0.0,
            avgNShow = bannersStat.avgNShow ?: 0.0,
            avgNShowComplete = bannersStat.avgNShowComplete ?: 0.0,
            conversions = 0,
            installs = 0,
            postViewConversions = 0,
            postViewInstalls = 0,
        )
    }
}
