package ru.yandex.direct.web.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.canvas.client.model.html5.Html5BatchResponse
import ru.yandex.direct.canvas.client.model.html5.Html5SourceResponse
import ru.yandex.direct.canvas.client.model.html5.Html5Tag
import ru.yandex.direct.core.entity.uac.model.CreativeType
import ru.yandex.direct.core.entity.uac.model.FileData
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.uac.service.UacCanvasService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.result.Result
import java.util.*

@Service
class UacHtml5ContentUploader(
    private val uacCanvasService: UacCanvasService
) {
    fun upload(
        subjectUser: User, accountId: String,
        fileData: FileData, creativeType: CreativeType, locale: Locale?,
        html5Tag: Html5Tag?
    ): Result<UacYdbContent> {
        val createSourceResult = uacCanvasService.uploadHtml5(subjectUser.clientId.asLong(), fileData, locale, html5Tag)
        if (!createSourceResult.isSuccessful) {
            return Result.broken(createSourceResult.validationResult)
        }
        val html5SourceResponse = createSourceResult.result!!

        val createCreativeResult = uacCanvasService.createHtml5Creative(
            subjectUser.clientId.asLong(), null, html5SourceResponse.id
        )
        if (!createCreativeResult.isSuccessful) {
            return Result.broken(createCreativeResult.validationResult)
        }
        val html5BatchResponse = createCreativeResult.result!!

        val uploadToDirectResult = uacCanvasService.uploadHtml5CreativeToDirect(
            subjectUser.clientId.asLong(), subjectUser.id, html5BatchResponse.id, html5BatchResponse.creatives[0].id
        )
        if (!uploadToDirectResult.isSuccessful) {
            return Result.broken(uploadToDirectResult.validationResult)
        }

        return Result.successful(
            fillUacYdbContent(accountId, creativeType, fileData, html5SourceResponse, html5BatchResponse)
        )
    }

    private fun fillUacYdbContent(
        accountId: String, creativeType: CreativeType, fileData: FileData,
        html5SourceResponse: Html5SourceResponse, html5BatchResponse: Html5BatchResponse
    ): UacYdbContent {
        return UacYdbContent(
            ownerId = accountId,
            type = MediaType.HTML5,
            thumb = html5SourceResponse.screenshotUrl ?: "",
            sourceUrl = html5SourceResponse.url ?: "",
            mdsUrl = html5SourceResponse.url,
            meta = fillContentMeta(creativeType, html5SourceResponse, html5BatchResponse),
            videoDuration = null,
            filename = fileData.name,
            accountId = accountId,
            directImageHash = null,
        )
    }

    private fun fillContentMeta(
        creativeType: CreativeType,
        html5SourceResponse: Html5SourceResponse,
        html5BatchResponse: Html5BatchResponse,
    ): Map<String, Any?> {
        return mapOf(
            "id" to html5SourceResponse.id,
            "screenshot_url" to html5SourceResponse.screenshotUrl,
            "preview_url" to html5SourceResponse.previewUrl,
            "url" to html5SourceResponse.url,
            "creative_type" to creativeType.getType(),
            "creative_id" to html5BatchResponse.creatives[0].id,
            "orig-size" to mapOf(
                "width" to html5SourceResponse.width,
                "height" to html5SourceResponse.height,
                "originalWidth" to html5SourceResponse.originalWidth,
                "originalHeight" to html5SourceResponse.originalHeight,
            )
        )
    }
}
