package ru.yandex.direct.web.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgeneration.ZenDomain
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService
import ru.yandex.direct.core.entity.campaign.service.WalletService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter
import ru.yandex.direct.core.entity.feed.model.UpdateStatus
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService
import ru.yandex.direct.core.entity.uac.service.UacAppInfoService
import ru.yandex.direct.core.entity.uac.validation.invalidAppUrl
import ru.yandex.direct.core.entity.uac.validation.invalidUrl
import ru.yandex.direct.core.entity.uac.validation.trackingUrlWrongRedirect
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.result.Result
import ru.yandex.direct.utils.UrlUtils
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.StringConstraints.notBlank
import ru.yandex.direct.validation.constraint.StringConstraints.validHref
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.web.entity.uac.model.links.LinkInfo
import ru.yandex.direct.web.entity.uac.model.links.successfulResult
import java.net.URLDecoder
import java.nio.charset.StandardCharsets
import ru.yandex.direct.core.entity.uac.model.MarketplaceInfo
import ru.yandex.direct.core.entity.uac.service.shopinshop.ShopInShopBusinessesService

@Service
class UacLinkService(
    private val bannerUrlCheckService: BannerUrlCheckService,
    private val uacAppInfoService: UacAppInfoService,
    private val ecomDomainsService: EcomDomainsService,
    private val feedService: FeedService,
    private val walletService: WalletService,
    private val uacMobileAppService: UacMobileAppService,
    private val featureService: FeatureService,
    private val shopInShopBusinessesService: ShopInShopBusinessesService
) {

    private fun validateUrl(url: String?): ValidationResult<String?, Defect<*>> {
        return validateObject(url) {
            check(notNull())
            check(notBlank())
            check(validHref())
        }
    }

    fun getLinkInfo(url: String, clientId: ClientId, subjectUid: Long, platform: Platform): Result<LinkInfo> {
        val validationResult = validateUrl(url)
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult)
        }

        val urlCheckResult = bannerUrlCheckService.checkUrls(listOf(url))
        val noError = urlCheckResult.result.all { it.isSuccessful && it.result.result }
        if (!urlCheckResult.isSuccessful || !noError) {
            return Result.broken(ValidationResult.failed(url, invalidUrl()))
        }

        val redirectResult = bannerUrlCheckService.getRedirect(url)
        if (!redirectResult.isSuccessful) {
            return Result.broken(ValidationResult.failed(url, trackingUrlWrongRedirect()))
        }

        val isZen = ZenDomain.isZenDomain(url)
        var appInfoValidation = getAppInfoValidationFromUrl(redirectResult.redirectUrl)
        if (appInfoValidation != null && appInfoValidation.hasAnyErrors()) {
            return Result.broken(appInfoValidation)
        }
        var appInfo = appInfoValidation?.value
        if (appInfo == null && featureService.isEnabledForClientId(clientId, FeatureName.VALIDATE_TRACKING_URLS_GOROTOR)) {
            val redirectUrl = uacMobileAppService.doValidateTrackingUrlRotor(url, platform)
            appInfoValidation = getAppInfoValidationFromUrl(redirectUrl)
            if (appInfoValidation != null && appInfoValidation.hasAnyErrors()) {
                return Result.broken(appInfoValidation)
            }
            appInfo = appInfoValidation?.value
        }

        val clientHasWallet = walletService.hasEnabledSharedWallet(subjectUid)
        if (!clientHasWallet) {
            return LinkInfo(isZen, appInfo).successfulResult()
        }

        val domain = ecomDomainsService.toMainMirrors(listOf(url))[url] ?: url

        val isEcomDomain =
            ecomDomainsService.getEcomDomainsByDomains(listOf(domain)).isNotEmpty()

        var isEcom = isEcomDomain || clientHasFeeds(clientId, ecomDomainsService.normalizeDomain(domain))

        val isMarketplace = ecomDomainsService.getMarketplacesByDomains(listOf(domain)).isNotEmpty()
        val marketplaceInfo = if (isMarketplace) {
            shopInShopBusinessesService.getBusinessInfoByUrl(url)?.let {
                // Ссылки на бизнесы маркетплейсов считаем екомовыми
                isEcom = true
                MarketplaceInfo(
                    additionalMarketplaceCommission = it.isAdditionalCommission,
                    suggestedUrlWithoutCommission = it.suggestedUrlWithoutCommission,
                    urlToShopInShopAccount = it.urlToShopInShopAccount
                )
            }
        } else null

        return LinkInfo(
            appInfo = appInfo,
            isEcom = isEcom,
            isMarketplace = isMarketplace,
            marketplaceInfo = marketplaceInfo,
            isZen = isZen
        ).successfulResult()
    }

    /**
     * Считаем ссылку екомовой, если у её владельца есть хотя бы один RETAIL фид,
     * даже если поисковый робот не распознал сайт екомовым.
     */
    private fun clientHasFeeds(clientId: ClientId, domain: String): Boolean {
        val query = FeedQueryFilter
            .newBuilder()
            .withUpdateStatuses(setOf(UpdateStatus.DONE))
            .build()
        val feeds = feedService.getFeeds(clientId, query)

        return feeds.any { feed -> feed.targetDomain == domain }
    }

    private fun getAppInfoValidationFromUrl(redirectUrl: String?): ValidationResult<AppInfo, Defect<*>>? {
        if (redirectUrl == null) {
            return null
        }
        val redirectUrlAscii = UrlUtils.urlPathToAsciiIfCan(redirectUrl)
        val decodedUrl = UrlUtils.encodeUrlQueryIfCan(URLDecoder.decode(redirectUrlAscii, StandardCharsets.UTF_8))
        return uacAppInfoService.getTruncatedAppInfoByUrl(decodedUrl, invalidAppUrl())
    }
}
