package ru.yandex.direct.web.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsService
import ru.yandex.direct.core.entity.retargeting.model.ConditionType
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService
import ru.yandex.direct.core.entity.retargeting.service.uc.UcRetargetingConditionService
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRule
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoalType
import ru.yandex.direct.core.entity.uac.service.RmpCampaignService
import ru.yandex.direct.core.entity.uac.service.UacRetargetingService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.multitype.entity.LimitOffset
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.StringConstraints.notBlank
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.MappingPathNodeConverter
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.web.entity.retargetinglists.service.GetRetargetingGoalsService
import ru.yandex.direct.web.entity.retargetinglists.service.UpdateRetargetingConditionsService
import ru.yandex.direct.web.entity.uac.converter.UacRetargetingConverter.toRetargetingConditionWeb
import ru.yandex.direct.web.entity.uac.converter.UacRetargetingConverter.toUacRetargetingCondition
import ru.yandex.direct.web.entity.uac.model.RetargetingConditionRequest
import ru.yandex.direct.web.entity.uac.validation.UacRetargetingConditionRuleGoalValidator.Companion.METRIKA_TYPES
import ru.yandex.direct.web.entity.uac.validation.UacRetargetingConditionRuleValidator

@Service
class UacRetargetingConditionAddAndUpdateService(
    private val mobileAppGoalsService: MobileAppGoalsService,
    private val retargetingConditionService: RetargetingConditionService,
    private val updateRetargetingConditionsService: UpdateRetargetingConditionsService,
    private val ucRetargetingConditionService: UcRetargetingConditionService,
    private val getRetargetingGoalsService: GetRetargetingGoalsService,
    private val rmpCampaignService: RmpCampaignService,
) {

    companion object {
        const val MIN_INCLUDE_RULES = 1
        const val MAX_INCLUDE_RULES = 1
        const val MAX_EXCLUDE_RULES = 1
        private val RETARGETING_CONDITION_CONVERTER = MappingPathNodeConverter
            .builder("RetargetingCondition")
            .replace(RetargetingCondition.RULES.name(), UacRetargetingCondition::conditionRules.name)
            .replace(RetargetingCondition.NAME.name(), UacRetargetingCondition::name.name)
            .build()
        val RETARGETING_CONDITION_CONVERTER_PROVIDER: DefaultPathNodeConverterProvider =
            DefaultPathNodeConverterProvider.builder()
                .register(RetargetingCondition::class.java, RETARGETING_CONDITION_CONVERTER)
                .build()
    }

    fun addUacRetargetingCondition(
        subjectUser: User,
        request: RetargetingConditionRequest
    ): Result<UacRetargetingCondition> {
        val operation = UacRetargetingConditionAddOperation(subjectUser, request)
        val validationResult = operation.prepare()
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult)
        }

        return operation.apply()
    }

    fun updateUacRetargetingCondition(
        subjectUser: User,
        request: RetargetingConditionRequest,
        retargetingConditionId: Long,
    ): Result<UacRetargetingCondition> {
        val operation = UacRetargetingConditionUpdateOperation(subjectUser, request, retargetingConditionId)
        val validationResult = operation.prepare()
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult)
        }

        return operation.apply()
    }

    abstract inner class UacRetargetingConditionOperation(
        private val subjectUser: User,
        private val request: RetargetingConditionRequest,
    ) {
        protected var prepared = false
        protected var existingGoalIdToName: Map<Long, String> = mapOf()
        protected var lalGoals: List<Goal> = listOf()
        protected var metrikaSegmentIdToGoal: Map<Long, Goal> = mapOf()

        fun prepare(): ValidationResult<RetargetingConditionRequest, Defect<*>> {
            check(!prepared)
            prepared = true

            val mobileApps = if (request.mobileAppId != null) rmpCampaignService.getMobileAppsByAppId(
                subjectUser.clientId,
                request.mobileAppId
            ) else listOf()
            val availableGoals = if (mobileApps.isEmpty()) {
                listOf()
            } else {
                mobileAppGoalsService.getGoalsByApps(subjectUser.clientId, mobileApps)
            }
            val availableGoalIds = availableGoals
                .map { it.id }
                .toSet()
            existingGoalIdToName = availableGoals.associate { it.id to it.name }

            val availableAudienceGoalIds = getRetargetingGoalsService
                .getMetrikaGoalsForRetargetingByGoalTypes(setOf(GoalType.AUDIENCE))
                .map { it.id }
                .toSet()
            val metrikaSegmentIds = request.conditionRules.flatMap {
                it.goals
                    .filter { goal -> goal.type in METRIKA_TYPES }
                    .map { goal -> goal.id }
            }.toSet()

            val allMetrikaGoals = retargetingConditionService.getMetrikaGoalsForRetargetingByIds(subjectUser.clientId,
                    (metrikaSegmentIds + availableAudienceGoalIds).toSet() )
            val availableAudienceGoals = allMetrikaGoals.filter { it.id in availableAudienceGoalIds }
            val metrikaSegmentGoals = allMetrikaGoals.filter { it.id in metrikaSegmentIds }

            metrikaSegmentIdToGoal = metrikaSegmentGoals
                .associateBy { it.id }
            val existingMetrikaSegmentIds = getRetargetingGoalsService
                .getMetrikaGoalsForRetargetingByGoalTypes(setOf(GoalType.SEGMENT))
                .map { it.id }
                .toSet()


            val lalGoalIds = request.conditionRules.flatMap {
                it.goals
                    .filter { goal ->
                        goal.type in listOf(
                            UacRetargetingConditionRuleGoalType.LAL,
                            UacRetargetingConditionRuleGoalType.LAL_AUDIENCE,
                            UacRetargetingConditionRuleGoalType.LAL_SEGMENT,
                        )
                    }
                    .map { goal -> goal.id }
            }.toSet()

            //объединение всех доступных целей
            val availableGoalsWithAudience = availableGoals + availableAudienceGoals.filter { it.id !in availableGoalIds }
            val availableGoalsWithAudienceIds = availableGoalsWithAudience.map { it.id }.toSet()
            val allAvailableGoals =
                availableGoalsWithAudience +
                    metrikaSegmentGoals.filter { it.id !in availableGoalsWithAudienceIds }

            lalGoals = ucRetargetingConditionService
                .getOrCreateLalRetargetingGoals(subjectUser.clientId, lalGoalIds, allAvailableGoals)
            val includeRulesCount = request.conditionRules
                .filter {
                    it.type in listOf(
                        UacRetargetingConditionRule.RuleType.OR,
                        UacRetargetingConditionRule.RuleType.ALL
                    )
                }
                .size
            val excludeRulesCount = request.conditionRules
                .filter { it.type == UacRetargetingConditionRule.RuleType.NOT }
                .size

            return validateObject(request) {
                property(RetargetingConditionRequest::name) {
                    check(notNull())
                    check(notBlank())
                }
                listProperty(RetargetingConditionRequest::conditionRules) {
                    check(CollectionConstraints.notEmptyCollection())
                    checkEachBy(
                        UacRetargetingConditionRuleValidator(
                            existingGoalIdToName,
                            availableAudienceGoalIds,
                            existingMetrikaSegmentIds
                        )
                    )
                    check(fromPredicate({ includeRulesCount in MIN_INCLUDE_RULES..MAX_INCLUDE_RULES }, invalidValue()))
                    check(fromPredicate({ excludeRulesCount <= MAX_EXCLUDE_RULES }, invalidValue()))
                }
            }
        }

        abstract fun apply(): Result<UacRetargetingCondition>
    }

    inner class UacRetargetingConditionAddOperation(
        private val subjectUser: User,
        private val request: RetargetingConditionRequest,
    ) : UacRetargetingConditionOperation(subjectUser, request) {
        private var applied = false

        override fun apply(): Result<UacRetargetingCondition> {
            check(prepared) { "Operation is not prepared" }
            check(!applied) { "Operation was already applied" }
            applied = true

            val retargetingCondition = UacRetargetingCondition(
                conditionRules = request.conditionRules,
                name = request.name
            )
            val directRetargetingCondition = UacRetargetingService.toCoreRetargetingCondition(
                retargetingCondition,
                subjectUser.clientId.asLong(),
                type = ConditionType.metrika_goals,
                parentIdToLalSegmentId = lalGoals.associate { it.parentId to it.id },
            )
            val retargetingConditionResult = retargetingConditionService.addRetargetingConditions(
                listOf(directRetargetingCondition), subjectUser.clientId
            )
            if (retargetingConditionResult.validationResult.hasAnyErrors()) {
                return Result.broken(retargetingConditionResult.validationResult)
            }
            val createdRetargetingCondition = retargetingConditionService.getRetargetingConditions(
                subjectUser.clientId, listOf(retargetingConditionResult.get(0).result), LimitOffset.maxLimited()
            ).firstOrNull()!!
            return Result.successful(
                toUacRetargetingCondition(
                    createdRetargetingCondition,
                    existingGoalIdToName,
                    lalGoals.associate { it.id to it.parentId },
                    metrikaSegmentIdToGoal,
                )
            )
        }
    }

    inner class UacRetargetingConditionUpdateOperation(
        private val subjectUser: User,
        private val request: RetargetingConditionRequest,
        private val retargetingConditionId: Long,
    ) : UacRetargetingConditionOperation(subjectUser, request) {
        private var applied = false

        override fun apply(): Result<UacRetargetingCondition> {
            check(prepared) { "Operation is not prepared" }
            check(!applied) { "Operation was already applied" }
            applied = true

            val retargetingConditionWeb = toRetargetingConditionWeb(
                request,
                subjectUser.clientId.asLong(),
                retargetingConditionId,
                parentIdToLalSegmentId = lalGoals.associate { it.parentId to it.id }
            )
            val retargetingConditionUpdate = updateRetargetingConditionsService
                .updateUacCondition(retargetingConditionWeb, subjectUser.clientId)
            if (retargetingConditionUpdate.hasAnyErrors()) {
                return Result.broken(retargetingConditionUpdate)
            }
            val retargetingCondition = retargetingConditionService.getRetargetingConditions(
                subjectUser.clientId, listOf(retargetingConditionId), LimitOffset.maxLimited()
            ).firstOrNull()!!
            return Result.successful(
                toUacRetargetingCondition(
                    retargetingCondition,
                    existingGoalIdToName,
                    lalGoals.associate { it.id to it.parentId },
                    metrikaSegmentIdToGoal,
                )
            )
        }
    }
}
