package ru.yandex.direct.web.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple
import ru.yandex.direct.core.entity.campaign.model.CampaignType.TEXT
import ru.yandex.direct.core.entity.crypta.service.CryptaSuggestService
import ru.yandex.direct.core.entity.feature.service.enabled
import ru.yandex.direct.core.entity.retargeting.model.ConditionType
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRule
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoal
import ru.yandex.direct.core.entity.uac.service.UacRetargetingService
import ru.yandex.direct.feature.FeatureName.UC_CUSTOM_AUDIENCE_ENABLED
import ru.yandex.direct.grid.processing.model.goal.GdGoalTruncated
import ru.yandex.direct.grid.processing.model.goal.GdGoalType
import ru.yandex.direct.grid.processing.service.goal.CryptaGoalsConverter

@Service
class UacRetargetingConditionService(
    private val retargetingConditionRepository: RetargetingConditionRepository,
    private val cryptaSuggestService: CryptaSuggestService,
    private val cryptaGoalsConverter: CryptaGoalsConverter,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(this::class.java)
    }

    /**
     * Функция собирает следующие:
     * - данные о retargeting_conditions группы (id, name)
     * - данные о name, type, description крипто-целей (с типом interests и host)
     * Далее заполняет недостающими данными итоговую модель UacRetargetingCondition для выдачи на фронт
     * Повторяет логику из профи
     * [ru.yandex.direct.grid.processing.service.showcondition.retargeting.RetargetingDataService.enhanceGoalsForCustomAudience]
     * Выполняется только для UC с включенной фичей uc_custom_audience_enabled
     */
    fun fillRetargetingCondition(
        shard: Int,
        campaign: CampaignSimple?,
        retargetingCondition: UacRetargetingCondition?,
    ): UacRetargetingCondition? {
        if (retargetingCondition == null || TEXT != campaign?.type || !UC_CUSTOM_AUDIENCE_ENABLED.enabled()) {
            return retargetingCondition
        }

        // Получаем данные (id,name) по условию показа группы. Если нет - берем их из грутовой заявки
        val retConditionNameByIds = retargetingConditionRepository
            .getRetargetingConditionNameByIds(shard, campaign.id, ConditionType.interests)
        if ((retConditionNameByIds?.size ?: 0) > 1) {
            logger.error("There can only be one retargeting condition in UC campaign, but ${campaign.id} " +
                "has ${retConditionNameByIds?.size}")
        }
        val retConditionIdAndName = retConditionNameByIds
            .map { Pair(it.key, it.value) }
            .lastOrNull()

        val interestGoalIds = retargetingCondition.conditionRules
            .flatMap { it.goals }
            .mapNotNull { it.id }

        // Собираем крипто-цели
        val customAudienceGoalByIds = cryptaSuggestService.getGoalsByIds(interestGoalIds)

        // Получаем текста (name,description) для целей
        val goalDataByIds = cryptaGoalsConverter
            .convertToTranslatedGdGoal(customAudienceGoalByIds.values)
            .associateBy { it.id }

        // Клонируем первоначальную модель с добавлением собранных данных по целям и условию
        return getCopyOfRetargetingCondition(
            retargetingCondition,
            retConditionIdAndName?.first ?: retargetingCondition.id,
            retConditionIdAndName?.second ?: retargetingCondition.name,
            goalDataByIds
        )
    }

    /**
     * Возвращает копию [retCondition] с изменениями в переданных параметрах
     */
    private fun getCopyOfRetargetingCondition(
        retCondition: UacRetargetingCondition,
        retConditionId: Long?,
        retConditionName: String?,
        goalDataByIds: Map<Long, GdGoalTruncated>,
    ): UacRetargetingCondition = retCondition.copy(
        id = retConditionId,
        name = retConditionName,
        conditionRules = getCopyOfRetargetingConditionRules(retCondition.conditionRules, goalDataByIds)
    )

    /**
     * Возвращает копию [retConditionRules] с изменениями в целях по переданным параметрам
     */
    private fun getCopyOfRetargetingConditionRules(
        retConditionRules: List<UacRetargetingConditionRule>,
        goalDataByIds: Map<Long, GdGoalTruncated>,
    ): List<UacRetargetingConditionRule> = retConditionRules
        .map { rule ->
            rule.copy(
                goals = getCopyOfRetargetingConditionGoals(rule.goals, goalDataByIds)
            )
        }

    /**
     * Возвращает копию списка целей [retConditionRuleGoals] с изменениями в переданных параметрах из [goalDataByIds]
     */
    private fun getCopyOfRetargetingConditionGoals(
        retConditionRuleGoals: List<UacRetargetingConditionRuleGoal>,
        goalDataByIds: Map<Long, GdGoalTruncated>,
    ): List<UacRetargetingConditionRuleGoal> = retConditionRuleGoals
        .map { goal ->
            val goalData = goalDataByIds[goal.id]
            goal.copy(
                name = goalData?.name ?: goal.name,
                type = UacRetargetingService.toUcGoalType(GdGoalType.toSource(goalData?.type)) ?: goal.type,
                description = goalData?.description ?: goal.description,
            )
        }
}
