package ru.yandex.direct.web.entity.uac.service

import org.asynchttpclient.AsyncHttpClient
import org.springframework.stereotype.Service
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.http.smart.annotations.Json
import ru.yandex.direct.http.smart.core.CallsPack
import ru.yandex.direct.http.smart.core.Smart
import ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors
import ru.yandex.direct.http.smart.http.GET
import ru.yandex.direct.http.smart.http.Headers
import ru.yandex.direct.http.smart.http.PartibleQuery;
import ru.yandex.direct.http.smart.http.Query
import ru.yandex.direct.web.entity.uac.model.SaasRequest
import ru.yandex.direct.web.entity.uac.model.SaasResponse
import java.time.Duration

@Service
class UacSaasClient(
    asyncHttpClient: AsyncHttpClient,
    config: DirectConfig,
) {
    private val baseUrl = config.getBranch("saas_client").getString("base_url")
    private val api: SaasApi = Smart.builder()
        .withParallelFetcherFactory(
            ParallelFetcherFactory(
                asyncHttpClient,
                FetcherSettings()
                    .withRequestRetries(3)
                    .withSoftTimeout(Duration.ofSeconds(2))
                    .withRequestTimeout(Duration.ofSeconds(3))
                    .withParallel(2)
            ))
        .withProfileName("saas_client")
        .withBaseUrl(baseUrl)
        .build()
        .create(SaasApi::class.java)

    fun suggest(requests: List<SaasRequest>, limit: Int): List<SaasResponse> {
        return api.getSuggest(
            "rmp_apps",
            requests.map { makeSuggestQuery(it.text, it.store) },
            "proto",
            "json",
            "1",
            makePruncount(limit),
            limit.toString(),
        ).executeCall()
    }

    private fun makePruncount(limit: Int): String {
        // pruncount отвечает за то, какой топ элементов после фильтрации на базовом поиске
        // попадет в формулу ранжирования.
        // Значение подобрано руками
        return "pruncount${limit * 5}"
    }

    private fun makeSuggestQuery(text: String?, store: String?): String {
        // Включение режима объектного саджеста в SAAS происходит через * в конце запроса
        // SAAS требует хотя бы один символ, чтобы саджест работал
        var textParam = text?.trim()
        if (textParam.isNullOrEmpty()) {
            textParam = "a"
        }
        // Явно включаем поиск только по заголовку и объектный саджест
        // Оборачиваем запрос в кавычки, чтобы поддерживать многословные запросы
        // Оставляем только приложения с российским регионом
        var query = "s_region:ru z_title:\"${textParam}*\""
        if (store != null) {
            query = "s_store:${store} " + query
        }
        return query
    }

    private fun <T> CallsPack<T>.executeCall(): List<T> {
        val results = execute(1).values
        for (result in results) {
            checkResultForErrors(result, { RuntimeException(it) })
        }
        return results.map { it!!.success }
    }
}

interface SaasApi {
    @GET("/")
    @Json
    @Headers("Content-Type: application/json")
    fun getSuggest(
        @Query("service") service: String,
        @PartibleQuery("text") text: List<String>,
        @Query("ms") ms: String,
        @Query("hr") hr: String,
        @Query("kps") kps: String,
        @Query("pron") pron: String,
        @Query("numdoc") numdoc: String,
        ): CallsPack<SaasResponse>
}
