package ru.yandex.direct.web.entity.uac.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.uac.model.TrackingUrl
import ru.yandex.direct.core.entity.uac.service.trackingurl.ParserType
import ru.yandex.direct.core.entity.uac.service.trackingurl.TrackingUrlParseService
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.web.entity.uac.model.ValidateTrackingUrlRequest
import ru.yandex.direct.web.entity.uac.model.ValidationType

@Service
class UacTrackingUrlValidationService(
    private val uacMobileAppService: UacMobileAppService,
    private val trackingUrlParseService: TrackingUrlParseService,
){

    fun validateAndParseTrackingUrl(
        validateTrackingUrlRequest: ValidateTrackingUrlRequest
    ): Pair<Defect<Void>?, TrackingUrl?> {
        val appValidationInfo: AppValidationInfo? = getAppValidationInfo(validateTrackingUrlRequest)

        return when(validateTrackingUrlRequest.validationType) {
            ValidationType.REDIRECT_URL -> validateAndParseRedirectUrl(
                validateTrackingUrlRequest.url, appValidationInfo
            )
            ValidationType.TRACKING_URL -> validateAndParseTrackingUrl(
                validateTrackingUrlRequest.url, appValidationInfo
            )
            ValidationType.IMPRESSION_URL -> validateAndParseImpressionUrl(
                validateTrackingUrlRequest.url, appValidationInfo
            )
        }
    }

    private fun getAppValidationInfo(validateTrackingUrlRequest: ValidateTrackingUrlRequest): AppValidationInfo? {
        if (validateTrackingUrlRequest.appInfoId != null) {
            return uacMobileAppService.getAppValidationInfoByAppInfoId(validateTrackingUrlRequest.appInfoId)
        }
        if (validateTrackingUrlRequest.appId != null) {
            return uacMobileAppService.getAppValidationInfoByAppId(validateTrackingUrlRequest.appId)
        }
        return null
    }

    private fun validateAndParseRedirectUrl(
        url: String,
        appValidationInfo: AppValidationInfo?
    ): Pair<Defect<Void>?, TrackingUrl?> {
        val defect = uacMobileAppService.validateRedirectUrl(url, appValidationInfo)
        return if (defect != null) {
            defect to null
        } else {
            null to trackingUrlParseService.parse(url, appValidationInfo?.platform, ParserType.TRACKING_URL)
        }
    }

    private fun validateAndParseTrackingUrl(
        url: String,
        appValidationInfo: AppValidationInfo?
    ): Pair<Defect<Void>?, TrackingUrl?> {
        val defect = uacMobileAppService.validateTrackingUrl(url, appValidationInfo)
        return if (defect != null) {
            defect to null
        } else {
            null to trackingUrlParseService.parse(url, appValidationInfo?.platform, ParserType.TRACKING_URL)
        }
    }

    private fun validateAndParseImpressionUrl(
        url: String,
        appValidationInfo: AppValidationInfo?
    ): Pair<Defect<Void>?, TrackingUrl?> {
        val defect = uacMobileAppService.validateImpressionUrl(url)
        return if (defect != null) {
            defect to null
        } else {
            null to trackingUrlParseService.parse(url, appValidationInfo?.platform, ParserType.IMPRESSION_URL)
        }
    }
}
