package ru.yandex.direct.web.entity.uac.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.audience.client.YaAudienceClient
import ru.yandex.direct.audience.client.exception.YaAudienceClientException
import ru.yandex.direct.audience.client.exception.YaAudienceClientTypedException
import ru.yandex.direct.audience.client.model.AudienceSegment
import ru.yandex.direct.audience.client.model.SegmentStatus
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.result.Result
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.web.entity.uac.validation.segmentCreationFromFileFailed
import ru.yandex.direct.web.entity.uac.validation.yaAudienceTypedClientError

@Service
class UacUploadAudienceSegmentFileService(
    private val yaAudienceClient: YaAudienceClient
) {
    private companion object {
        private val logger = LoggerFactory.getLogger(UacUploadAudienceSegmentFileService::class.java)
    }

    fun uploadAudienceSegmentFile(
        subjectUser: User,
        file: MultipartFile,
        segmentName: String,
        lang: String?
    ): Result<AudienceSegment> = Trace.current().profile("uac:uploadAudienceSegmentFile")
        .use {
            return try {
                val audienceSegment = yaAudienceClient.uploadSegment(subjectUser.login, file.bytes)
                val confirmedSegment = yaAudienceClient.confirmYuidSegment(
                    subjectUser.login,
                    audienceSegment.id,
                    segmentName,
                    lang
                )
                Result.successful(confirmedSegment)
            } catch (ex: YaAudienceClientException) {
                logger.error(ex.message, ex)
                when (ex) {
                    is YaAudienceClientTypedException ->
                        Result.broken(ValidationResult.failed(file.name, yaAudienceTypedClientError(ex.message)))
                    else -> Result.broken(ValidationResult.failed(file.name, segmentCreationFromFileFailed()))
                }
            }
        }

    fun getRetargetingSegmentsByStatus(subjectUser: User, status: SegmentStatus): Result<List<AudienceSegment>> {
        return try {
            val segments = yaAudienceClient.getSegments(subjectUser.login)
                .filter { status == it.status }
            Result.successful(segments)
        } catch (ex: YaAudienceClientException) {
            logger.error(ex.message, ex)
            Result.broken(ValidationResult.failed(status, segmentCreationFromFileFailed()))
        }
    }
}
