package ru.yandex.direct.web.entity.uac.service

import java.util.Locale
import java.util.Optional
import org.springframework.stereotype.Service
import ru.yandex.direct.canvas.client.model.video.AdditionResponse
import ru.yandex.direct.canvas.client.model.video.VideoUploadResponse
import ru.yandex.direct.core.entity.uac.UacCommonUtils.mapFromJson
import ru.yandex.direct.core.entity.uac.model.CreativeType
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbContent
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.result.Result
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.defect.CollectionDefects
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.core.entity.uac.getAvatarsThumbSize
import ru.yandex.direct.core.entity.uac.model.FileData
import ru.yandex.direct.core.entity.uac.service.UacAvatarsService
import ru.yandex.direct.core.entity.uac.service.UacCanvasService

@Service
class UacVideoContentUploader(
    private val uacAvatarsService: UacAvatarsService,
    private val uacCanvasService: UacCanvasService,
) {
    companion object {
        private const val VIDEO_MAX_SIZE = 100 * 1024 * 1024
    }

    fun uploadFromFile(
        subjectUser: User, accountId: String,
        file: FileData, creativeType: CreativeType, locale: Locale?
    ): Result<UacYdbContent> {
        return ByFileUploader(subjectUser, accountId, file, creativeType, locale).upload()
    }

    fun uploadFromUrl(
        subjectUser: User, accountId: String,
        url: String, sourceUrl: String?, creativeType: CreativeType, locale: Locale?
    ): Result<UacYdbContent> {
        return ByUrlUploader(subjectUser, accountId, url, sourceUrl, creativeType, locale).upload()
    }

    abstract inner class BaseUploader(
        private val subjectUser: User,
        private val accountId: String,
        private val creativeType: CreativeType,
        private val locale: Locale?,
    ) {
        private lateinit var videoUploadResponse: VideoUploadResponse
        private var additionResponse: AdditionResponse? = null
        private lateinit var thumbUrl: String

        private val validCreativeTypes = listOf(
            CreativeType.RMP, CreativeType.CPM, CreativeType.TGO, CreativeType.NON_SKIPPABLE_CPM
        )

        fun upload(): Result<UacYdbContent> {
            val creativeTypeValidationResult = validateCreativeType()
            if (creativeTypeValidationResult.hasAnyErrors()) {
                return Result.broken(creativeTypeValidationResult)
            }

            val uploadToCanvasResult = uploadVideoToCanvas()
            if (!uploadToCanvasResult.isSuccessful) {
                return Result.broken(uploadToCanvasResult.validationResult)
            }
            videoUploadResponse = uploadToCanvasResult.result

            if (allowedToCreateCreative()) {
                val createCreativeResult = createCreative()
                if (!createCreativeResult.isSuccessful) {
                    return Result.broken(createCreativeResult.validationResult)
                }
                additionResponse = createCreativeResult.result
            }

            thumbUrl = uploadThumbToAvatars()

            return Result.successful(fillUacYdbContent())
        }

        private fun validateCreativeType(): ValidationResult<CreativeType, Defect<*>> {
            if (!validCreativeTypes.contains(creativeType)) {
                return ValidationResult.failed(creativeType, CommonDefects.invalidValue())
            }
            return ValidationResult.success(creativeType)
        }

        protected abstract fun uploadVideoToCanvas(): Result<VideoUploadResponse>

        private fun allowedToCreateCreative(): Boolean {
            return videoUploadResponse.status == VideoUploadResponse.FileStatus.READY || videoUploadResponse.createEarlyCreative
        }

        private fun createCreative(): Result<AdditionResponse> {
            return uacCanvasService.createCreative(
                subjectUser.clientId.asLong(), videoUploadResponse.presetId, videoUploadResponse.id
            )
        }

        private fun uploadThumbToAvatars(): String {
            val uploadToAvatarsResult = uacAvatarsService.uploadImageByUrl(videoUploadResponse.thumbnailUrl, null)
            if (!uploadToAvatarsResult.isSuccessful) {
                return videoUploadResponse.thumbnailUrl
            }
            return uacAvatarsService.getReadUrl(
                uploadToAvatarsResult.result,
                getAvatarsThumbSize(uploadToAvatarsResult.result)
            )
        }

        private fun fillUacYdbContent(): UacYdbContent {
            return UacYdbContent(
                ownerId = accountId,
                type = MediaType.VIDEO,
                thumb = thumbUrl,
                sourceUrl = getSourceUrl() ?: videoUploadResponse.url,
                mdsUrl = videoUploadResponse.url,
                meta = fillContentMeta(),
                videoDuration = videoUploadResponse.duration?.toInt(),
                filename = videoUploadResponse.name,
                accountId = accountId,
                directImageHash = null,
            )
        }

        protected abstract fun getSourceUrl(): String?

        private fun fillContentMeta(): Map<String, Any?> {
            return mapOf(
                "creative_id" to additionResponse?.creativeId,
                "vast" to additionResponse?.vast,
                "is_tgo" to (creativeType == CreativeType.TGO),
                "creative_type" to creativeType.getType(),
                "status" to (if (additionResponse != null) "ready" else "converting"),
                "mime_type" to videoUploadResponse.mimeType,
                "preset_id" to videoUploadResponse.presetId,
                "id" to videoUploadResponse.id,
                "formats" to videoUploadResponse.formats
                    .map { mapFromJson(JsonUtils.toJson(it)) },
                "thumb" to mapFromJson(JsonUtils.toJson(videoUploadResponse.thumbnail)),
                "orig-size" to mapOf(
                    "width" to videoUploadResponse.width,
                    "height" to videoUploadResponse.height,
                )
            )
        }
    }

    inner class ByFileUploader(
        private val subjectUser: User,
        private val accountId: String,
        private val fileData: FileData,
        private val creativeType: CreativeType,
        private val locale: Locale?,
    ) : BaseUploader(subjectUser, accountId, creativeType, locale) {
        override fun getSourceUrl(): String? {
            return null
        }

        override fun uploadVideoToCanvas(): Result<VideoUploadResponse> {
            if (fileData.data.size > VIDEO_MAX_SIZE) {
                return Result.broken(
                    ValidationResult.failed(fileData, CollectionDefects.maxCollectionSize(VIDEO_MAX_SIZE))
                )
            }
            return uacCanvasService.uploadVideo(subjectUser.clientId.asLong(), fileData, creativeType, locale)
        }
    }

    inner class ByUrlUploader(
        private val subjectUser: User,
        private val accountId: String,
        private val url: String,
        private val sourceUrl: String?,
        private val creativeType: CreativeType,
        private val locale: Locale?,
    ) : BaseUploader(subjectUser, accountId, creativeType, locale) {
        override fun getSourceUrl(): String? {
            return sourceUrl
        }

        override fun uploadVideoToCanvas(): Result<VideoUploadResponse> {
            return uacCanvasService.uploadVideoByUrl(subjectUser.clientId.asLong(), url, creativeType, locale)
        }
    }
}
