package ru.yandex.direct.web.entity.uac.service

import java.math.BigDecimal
import java.util.EnumSet
import org.slf4j.Logger
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.campaign.model.TextCampaign
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects
import ru.yandex.direct.core.entity.campaign.service.validation.DisableDomainValidationService
import ru.yandex.direct.core.entity.client.service.ClientGeoService
import ru.yandex.direct.core.entity.client.service.ClientLimitsService
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.feature.service.enabled
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseConstraints.maxLengthKeywordsWithoutSpecSymbolsAndSpaces
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator.ValidationMode
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.minusphrase.MinusPhraseValidator.minusKeywordIsValid
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppAlternativeStore
import ru.yandex.direct.core.entity.performancefilter.schema.compiled.PerformanceDefault
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator
import ru.yandex.direct.core.entity.retargeting.service.validation2.AddRetargetingConditionValidationService2
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository
import ru.yandex.direct.core.entity.uac.UacCommonUtils
import ru.yandex.direct.core.entity.uac.converter.UacBidModifiersConverter.toUacAdjustments
import ru.yandex.direct.core.entity.uac.model.AdvType.CPM_BANNER
import ru.yandex.direct.core.entity.uac.model.AdvType.MOBILE_CONTENT
import ru.yandex.direct.core.entity.uac.model.AdvType.TEXT
import ru.yandex.direct.core.entity.uac.model.AltAppStore.Companion.toCoreType
import ru.yandex.direct.core.entity.uac.model.Content
import ru.yandex.direct.core.entity.uac.model.MediaType
import ru.yandex.direct.core.entity.uac.model.UacCampaignOptions
import ru.yandex.direct.core.entity.uac.model.UacStrategyName
import ru.yandex.direct.core.entity.uac.model.UacStrategyPlatform.SEARCH
import ru.yandex.direct.core.entity.uac.model.UpdateUacCampaignRequest
import ru.yandex.direct.core.entity.uac.model.relevance_match.UacRelevanceMatchCategory
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.moneyToDb
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils.toIdLong
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.AudienceSegmentsService
import ru.yandex.direct.core.entity.uac.service.BaseUacCampaignService
import ru.yandex.direct.core.entity.uac.service.BaseUacContentService
import ru.yandex.direct.core.entity.uac.service.CpmBannerCampaignService
import ru.yandex.direct.core.entity.uac.service.RmpCampaignService
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.entity.uac.service.UacCampaignsCoreService
import ru.yandex.direct.core.entity.uac.service.UacGeoService.getGeoForUacGroups
import ru.yandex.direct.core.entity.uac.validation.maxImageContentSize
import ru.yandex.direct.core.entity.uac.validation.maxVideoContentSize
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.grid.processing.service.campaign.uc.UcCampaignMutationService
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler
import ru.yandex.direct.model.KtModelChanges
import ru.yandex.direct.regions.GeoTree
import ru.yandex.direct.result.Result
import ru.yandex.direct.utils.mapToSet
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan
import ru.yandex.direct.validation.constraint.StringConstraints.notBlank
import ru.yandex.direct.validation.constraint.StringConstraints.validHref
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.defect.CommonDefects.objectNotFound
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.check
import ru.yandex.direct.validation.util.checkEach
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.web.entity.uac.converter.UacCampaignConverter
import ru.yandex.direct.web.entity.uac.converter.UacCampaignConverter.toCoreCpmCampaignModelChanges
import ru.yandex.direct.web.entity.uac.converter.UacTextCampaignConverter.toGdUpdateUcCampaignInput
import ru.yandex.direct.web.entity.uac.model.PatchCampaignInternalRequest
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_IMAGE_MAX_LENGTH
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_IMAGE_MAX_LENGTH_BY_FEATURE
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_TEXTS_MAX_LENGTH
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_TEXTS_MAX_LENGTH_BY_FEATURE
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_TITLES_MAX_LENGTH
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_TITLES_MAX_LENGTH_BY_FEATURE
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_VIDEO_MAX_LENGTH
import ru.yandex.direct.web.entity.uac.service.UacConstants.UC_VIDEO_MAX_LENGTH_BY_FEATURE
import ru.yandex.direct.web.entity.uac.validation.CpmAssetValidator
import ru.yandex.direct.web.entity.uac.validation.CpmImpressionLimitsValidator
import ru.yandex.direct.web.entity.uac.validation.CpmRetargetingsConditionsValidator
import ru.yandex.direct.web.entity.uac.validation.CpmSocdemValidator
import ru.yandex.direct.web.entity.uac.validation.UacDisabledPlacesValidator
import ru.yandex.direct.web.entity.uac.validation.UacFeedFiltersValidator
import ru.yandex.direct.web.entity.uac.validation.UacGoalsValidator
import ru.yandex.direct.web.entity.uac.validation.UacSearchLiftValidator
import ru.yandex.direct.web.entity.uac.validation.UcRetargetingConditionValidator

@Service
abstract class WebBaseUacCampaignUpdateService(
    private val uacBannerService: UacBannerService,
    private val uacMobileAppService: UacMobileAppService,
    private val uacModifyCampaignDataContainerFactory: UacModifyCampaignDataContainerFactory,
    private val rmpCampaignService: RmpCampaignService,
    private val uacCampaignsCoreService: UacCampaignsCoreService,
    private val clientService: ClientService,
    private val ucCampaignMutationService: UcCampaignMutationService,
    private val uacCampaignValidationService: UacCampaignValidationService,
    private val uacGoalsService: UacGoalsService,
    private val clientGeoService: ClientGeoService,
    private val cpmBannerCampaignService: CpmBannerCampaignService,
    private val uacAdjustmentsService: UacAdjustmentsService,
    private val baseUacContentService: BaseUacContentService,
    private val uacPropertiesService: UacPropertiesService,
    private val baseUacCampaignService: BaseUacCampaignService,
    private val sspPlatformsRepository: SspPlatformsRepository,
    private val hostingsHandler: HostingsHandler,
    private val bannerImageFormatRepository: BannerImageFormatRepository,
    private val shardHelper: ShardHelper,
    private val cpmBannerService: UacCpmBannerService,
    private val clientLimitsService: ClientLimitsService,
    private val disableDomainValidationService: DisableDomainValidationService,
    private val rmpStrategyValidatorFactory: RmpStrategyValidatorFactory,
    private val featureService: FeatureService,
    private val retargetingConditionValidationService: AddRetargetingConditionValidationService2,
    private val audienceSegmentsService: AudienceSegmentsService,
    private val feedService: FeedService,
    private val filterSchemaStorage: PerformanceFilterStorage,
) {

    abstract fun getLogger(): Logger

    fun updateCampaign(
        uacCampaign: UacYdbCampaign,
        directCampaignId: Long,
        request: PatchCampaignInternalRequest,
        operator: User,
        client: User,
        multipleAdsInUc: Boolean = false,
        updateMinusKeywords: Boolean = false,
    ): Result<UacYdbCampaign> {
        val operation =
            UacCampaignUpdateOperation(
                uacCampaign,
                directCampaignId,
                request,
                operator,
                client,
                multipleAdsInUc,
                updateMinusKeywords
            )

        val validationResult = operation.prepare()
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult)
        }

        return operation.apply()
    }

    inner class UacCampaignUpdateOperation(
        private val uacYdbCampaign: UacYdbCampaign,
        private val directCampaignId: Long,
        private val request: PatchCampaignInternalRequest,
        private val operator: User,
        private val client: User,
        private val multipleAdsInUc: Boolean = false,
        private val updateMinusKeywords: Boolean = false,
    ) {

        private val clientId = client.clientId
        private val operatorUid = operator.uid
        private lateinit var contentById: Map<String, Content>

        private lateinit var clientGeoTree: GeoTree

        fun prepare(): ValidationResult<PatchCampaignInternalRequest, Defect<*>> {
            val appInfo = uacMobileAppService.getAppInfo(request.appId ?: uacYdbCampaign.appId)

            val contentIds = request.contentIds ?: emptyList()
            contentById = baseUacContentService.getContents(contentIds)
                .associateBy { it.id }

            clientGeoTree = clientGeoService.getClientTranslocalGeoTree(client.clientId)

            val relevanceMatchCategoriesAreAllowed = featureService
                .isEnabledForClientId(client.clientId, FeatureName.RELEVANCE_MATCH_CATEGORIES_ALLOWED_IN_UC)
            val altAppStoresAreAllowed = featureService
                .isEnabledForClientId(client.clientId, FeatureName.ENABLE_ALTERNATIVE_STORES_IN_UAC)

            val ucMaxTitlesSize = if (multipleAdsInUc) UC_TITLES_MAX_LENGTH_BY_FEATURE else UC_TITLES_MAX_LENGTH
            val ucMaxTextsSize = if (multipleAdsInUc) UC_TEXTS_MAX_LENGTH_BY_FEATURE else UC_TEXTS_MAX_LENGTH
            val ucMaxImagesSize = if (multipleAdsInUc) UC_IMAGE_MAX_LENGTH_BY_FEATURE else UC_IMAGE_MAX_LENGTH
            val ucMaxVideosSize = if (multipleAdsInUc) UC_VIDEO_MAX_LENGTH_BY_FEATURE else UC_VIDEO_MAX_LENGTH

            val imageContentById = contentById!!.filterValues { it.type == MediaType.IMAGE }
            val videoContentById = contentById!!.filterValues { it.type == MediaType.VIDEO }
            val isOnlyPlayableContents = if (contentIds.isNotEmpty()) {
                contentById!!.filterValues { it.type != MediaType.HTML5 }.isEmpty()
            } else {
                baseUacCampaignService
                    .getMediaCampaignContentsForCampaign(uacYdbCampaign.id.toIdLong())
                    .none { it.type != MediaType.HTML5 }
            }
            val allowEmptyTexts =
                uacYdbCampaign.isDraft || (isOnlyPlayableContents && uacYdbCampaign.advType == MOBILE_CONTENT)

            val isSkadNetworkEnabled = uacYdbCampaign.skadNetworkEnabled ?: false

            return validateObject(request) {
                listProperty(PatchCampaignInternalRequest::titles) {
                    check(notEmptyCollection(), When.isTrue(!allowEmptyTexts))
                    checkEachBy(uacCampaignValidationService.getTitleValidator())
                    if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                        check(maxListSize(uacPropertiesService.maxTitles))
                    } else if (uacYdbCampaign.advType == TEXT) {
                        check(maxListSize(ucMaxTitlesSize))
                    } else if (uacYdbCampaign.advType == CPM_BANNER) {
                        check(CommonConstraints.isNull())
                    }
                }
                listProperty(PatchCampaignInternalRequest::texts) {
                    check(notEmptyCollection(), When.isTrue(!allowEmptyTexts))
                    checkEachBy(uacCampaignValidationService.getTextValidator(uacYdbCampaign.advType))
                    if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                        check(maxListSize(uacPropertiesService.maxTexts))
                    } else if (uacYdbCampaign.advType == TEXT) {
                        check(maxListSize(ucMaxTextsSize))
                    } else if (uacYdbCampaign.advType == CPM_BANNER) {
                        check(CommonConstraints.isNull())
                    }
                }
                listProperty(PatchCampaignInternalRequest::minusKeywords) {
                    if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                        checkBy(minusKeywordIsValid(ValidationMode.ONE_ERROR_PER_TYPE))
                        check(maxLengthKeywordsWithoutSpecSymbolsAndSpaces(CAMPAIGN_MINUS_KEYWORDS_MAX_LENGTH))
                    }
                }
                listProperty(PatchCampaignInternalRequest::contentIds) {
                    check(
                        notEmptyCollection(), When.isTrue(
                            !uacYdbCampaign.isDraft
                                || uacYdbCampaign.advType != MOBILE_CONTENT
                                || uacYdbCampaign.strategyPlatform != SEARCH
                        )
                    )

                    checkEach(objectNotFound()) { contentId ->
                        val content = contentById!![contentId]
                        content != null
                    }
                    checkEach(invalidValue()) { contentId ->
                        // todo отдельный дефект "видео не готово"?
                        val content = contentById!![contentId]
                        content?.type != MediaType.VIDEO || content.meta.containsKey("creative_id")
                    }
                }
                listProperty(PatchCampaignInternalRequest::adjustments) {
                    if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                        check(maxListSize(UacConstants.UAC_ADJUSTMENTS_MAX_LENGTH))
                    } else {
                        check(CommonConstraints.isNull(), When.isTrue(uacYdbCampaign.advType != MOBILE_CONTENT))
                    }
                    checkEach(invalidValue()) { adjustment -> uacAdjustmentsService.isValidAdjustment(adjustment) }
                }
                property(PatchCampaignInternalRequest::regions) {
                    checkBy(
                        { regions -> RegionIdsValidator().apply(regions, clientGeoTree) },
                        When.isTrue(request.regions != null && (request.regions.isNotEmpty() || !uacYdbCampaign.isDraft))
                    )
                }
                property(PatchCampaignInternalRequest::appId) {
                    check(objectNotFound()) {
                        appInfo != null
                    }
                }
                if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                    property(PatchCampaignInternalRequest::trackingUrl) {
                        check(notBlank(), When.isTrue(!uacYdbCampaign.isDraft))
                        check(
                            { trackingUrl -> uacMobileAppService.validateTrackingUrl(trackingUrl, appInfo) },
                            When.isValid()
                        )
                    }
                    property(PatchCampaignInternalRequest::impressionUrl) {
                        check(uacMobileAppService::validateImpressionUrl)
                    }
                } else {
                    property(PatchCampaignInternalRequest::trackingUrl) {
                        check(CommonConstraints.isNull())
                    }
                    property(PatchCampaignInternalRequest::impressionUrl) {
                        check(CommonConstraints.isNull())
                    }
                }
                if (uacYdbCampaign.advType != TEXT) {
                    property(PatchCampaignInternalRequest::zenPublisherId) {
                        check(CommonConstraints.isNull())
                    }
                }
                property(PatchCampaignInternalRequest::href) {
                    check(notBlank())
                    check(validHref())
                }
                property(PatchCampaignInternalRequest::contentIds) {
                    check(maxImageContentSize(ucMaxImagesSize), When.isTrue(uacYdbCampaign.advType == TEXT)) {
                        imageContentById.size <= ucMaxImagesSize
                    }
                    check(maxVideoContentSize(ucMaxVideosSize), When.isTrue(uacYdbCampaign.advType == TEXT)) {
                        videoContentById.size <= ucMaxVideosSize
                    }
                    check(
                        Constraint.fromPredicate(
                            { _ ->
                                contentById.values.mapNotNull { it.meta[UacCommonUtils.CREATIVE_TYPE_KEY] }
                                    .distinct().size == 1
                            },
                            invalidValue()
                        ),
                        When.isTrue(uacYdbCampaign.advType == CPM_BANNER && uacYdbCampaign.cpmAssets != null)
                    )

                }

                property(PatchCampaignInternalRequest::cpmAssets) {
                    if (uacYdbCampaign.advType == CPM_BANNER) {
                        check(
                            Constraint.fromPredicate({ cpmAssets ->
                                (contentById.keys - cpmAssets?.keys.orEmpty()).isEmpty()
                            }, invalidValue()),
                            When.notNull()
                        )

                        request.cpmAssets?.entries?.forEach {
                            item(it.value, it.key).checkBy(
                                CpmAssetValidator(
                                    uacCampaignValidationService,
                                    bannerImageFormatRepository,
                                    shardHelper,
                                    clientId
                                )
                            )
                        }

                    } else {
                        check(CommonConstraints.isNull())
                    }
                }

                property(PatchCampaignInternalRequest::uacBrandsafety) {
                    if (uacYdbCampaign.advType == CPM_BANNER) {
                    } else {
                        check(CommonConstraints.isNull())
                    }
                }
                val uacDisabledPlacesValidator = UacDisabledPlacesValidator(
                    clientLimitsService,
                    clientId,
                    hostingsHandler,
                    disableDomainValidationService,
                    sspPlatformsRepository
                )
                property(PatchCampaignInternalRequest::uacDisabledPlaces) {
                    if (uacYdbCampaign.advType != TEXT) {
                        checkBy(uacDisabledPlacesValidator, When.notNull())
                    } else {
                        check(CommonConstraints.isNull())
                    }
                }

                property(PatchCampaignInternalRequest::videosAreNonSkippable) {
                    if (uacYdbCampaign.advType == CPM_BANNER) {
                        check(CommonConstraints.notNull())
                    } else {
                        check(CommonConstraints.isNull())
                    }
                }
                property(PatchCampaignInternalRequest::brandSurveyId) {
                    if (uacYdbCampaign.advType != CPM_BANNER) {
                        check(CommonConstraints.isNull())
                    }
                }
                property(PatchCampaignInternalRequest::brandSurveyName) {
                    if (uacYdbCampaign.advType != CPM_BANNER) {
                        check(CommonConstraints.isNull())
                    }
                }
                property(PatchCampaignInternalRequest::searchLift) {
                    if (uacYdbCampaign.advType != CPM_BANNER) {
                        check(CommonConstraints.isNull())
                    } else {
                        val searchLiftEnabled = featureService
                            .isEnabledForClientId(client.clientId, FeatureName.SEARCH_LIFT)

                        check(CommonConstraints.isNull(), When.isFalse(searchLiftEnabled))
                        checkBy(UacSearchLiftValidator(), When.notNullAnd(When.isTrue(searchLiftEnabled)))
                    }
                }
                property(PatchCampaignInternalRequest::socdem) {
                    if (uacYdbCampaign.advType == CPM_BANNER) {
                        check(CommonConstraints.notNull())
                        checkBy(CpmSocdemValidator(), When.isValid())
                    }
                }
                property(PatchCampaignInternalRequest::goals) {
                    checkBy(UacGoalsValidator(request.cpa, request.crr), When.isTrue(uacYdbCampaign.advType == TEXT))
                }
                property(PatchCampaignInternalRequest::strategy) {
                    if (uacYdbCampaign.advType == CPM_BANNER) {
                        check(CommonConstraints.notNull())
                    } else if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                        val trackingUrl = uacMobileAppService.getTrackingUrl(
                            request.trackingUrl ?: uacYdbCampaign.trackingUrl,
                            appInfo
                        )
                        val availableStrategyGoals = uacGoalsService.getAvailableStrategyGoalsForRmp(
                            clientId,
                            trackingUrl,
                            appInfo,
                            isSkadNetworkEnabled
                        )
                        checkBy(
                            rmpStrategyValidatorFactory.createRmpStrategyValidator(
                                clientId,
                                availableStrategyGoals
                            ), When.notNull()
                        )
                    } else {
                        check(CommonConstraints.isNull())
                    }
                }
                property(PatchCampaignInternalRequest::targetId) {
                    if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                        val trackingUrl = uacMobileAppService.getTrackingUrl(
                            request.trackingUrl ?: uacYdbCampaign.trackingUrl,
                            appInfo
                        )
                        val avl = uacGoalsService.getAvailableStrategyGoalsForRmp(
                            clientId,
                            trackingUrl,
                            appInfo,
                            isSkadNetworkEnabled
                        )
                        check(Constraint.fromPredicate({
                            val strategies: Set<UacStrategyName>? = avl.goals[it]
                            strategies?.contains(request.strategy?.uacStrategyName) ?: false
                        }, invalidValue()), When.isTrue(request.strategy != null))
                    }
                }
                property(PatchCampaignInternalRequest::showsFrequencyLimit) {
                    if (uacYdbCampaign.advType != CPM_BANNER) {
                        check(CommonConstraints.isNull())
                    } else {
                        checkBy(CpmImpressionLimitsValidator(), When.notNull())
                    }
                }
                property(PatchCampaignInternalRequest::retargetingCondition) {
                    if (uacYdbCampaign.advType == TEXT) {
                        val ucCustomAudienceEnabled = featureService
                            .isEnabledForClientId(client.clientId, FeatureName.UC_CUSTOM_AUDIENCE_ENABLED)

                        val ucRetargetingConditionValidator = UcRetargetingConditionValidator(
                            retargetingConditionValidationService,
                            clientId,
                        )
                        check(CommonConstraints.isNull(), When.isFalse(ucCustomAudienceEnabled))
                        checkBy(ucRetargetingConditionValidator, When.notNullAnd(When.isTrue(ucCustomAudienceEnabled)))
                    } else if (uacYdbCampaign.advType == CPM_BANNER) {
                        checkBy(CpmRetargetingsConditionsValidator(), When.notNull())
                    } else if (uacYdbCampaign.advType == MOBILE_CONTENT) {
                        check(Constraint.fromPredicate({ it?.id != null }, invalidValue()), When.notNull())
                    }
                }

                /* не хотим оказаться в состоянии recommendationsManagementEnabled = false,
                 * priceRecommendationsManagementEnabled = true.
                 * И не хотим эти галочки взведёнными в товарных кампаниях
                 */
                if (uacYdbCampaign.isEcom == true) {
                    property(request::isRecommendationsManagementEnabled) {
                        check(
                            Constraint.fromPredicate(
                                { request.isRecommendationsManagementEnabled != true },
                                CommonDefects.inconsistentState()
                            )
                        )
                    }
                    property(request::isPriceRecommendationsManagementEnabled) {
                        check(
                            Constraint.fromPredicate(
                                { request.isPriceRecommendationsManagementEnabled != true },
                                CommonDefects.inconsistentState()
                            )
                        )
                    }
                } else if (request.isPriceRecommendationsManagementEnabled == true) {
                    property(request::isPriceRecommendationsManagementEnabled) {
                        val requestRecommendationsManagementEnabled = request.isRecommendationsManagementEnabled
                        check(
                            Constraint.fromPredicate(
                                {
                                    requestRecommendationsManagementEnabled == true ||
                                        (requestRecommendationsManagementEnabled == null
                                            && uacYdbCampaign.recommendationsManagementEnabled == true)
                                },
                                CommonDefects.inconsistentState()
                            ), When.isValid()
                        )
                    }
                } else if (request.isRecommendationsManagementEnabled == false) {
                    property(request::isRecommendationsManagementEnabled) {
                        val requestPriceRecommendationsManagementEnabled =
                            request.isPriceRecommendationsManagementEnabled
                        check(
                            Constraint.fromPredicate(
                                {
                                    requestPriceRecommendationsManagementEnabled == false ||
                                        (requestPriceRecommendationsManagementEnabled == null
                                            && uacYdbCampaign.priceRecommendationsManagementEnabled == false)
                                },
                                CommonDefects.inconsistentState()
                            )
                        )
                    }
                }

                if (request.isEcom != null && request.isEcom) {
                    val counterIdsLongList = request.counters?.map { it.toLong() } ?: listOf()
                    val availableCounters = uacGoalsService.getAvailableCounters(clientId, counterIdsLongList)
                    val goalsByIds = uacGoalsService.getAvailableGoalsForEcomCampaign(operatorUid, clientId, request)

                    listProperty(PatchCampaignInternalRequest::counters) {
                        check(CommonConstraints.notNull())
                        check(notEmptyCollection())
                        check({ counters ->
                            if (availableCounters.intersect(counters).isEmpty()) {
                                CampaignDefects.metrikaCounterIsUnavailable()
                            } else null
                        }, When.isValid())
                        checkEach(CommonConstraints.notNull())
                    }

                    property(PatchCampaignInternalRequest::feedId) {
                        check(CommonConstraints.notNull())
                        check(CommonConstraints.validId())
                    }

                    if (request.feedId != null) {
                        val feed = feedService.getFeedsSimple(
                            shardHelper.getShardByClientIdStrictly(clientId), listOf(request.feedId)).getOrNull(0)
                        val filterSchema = feed
                            ?.let { filterSchemaStorage.getFilterSchema(it.businessType, it.feedType) }
                            ?: PerformanceDefault()

                        val uacFeedFiltersValidator = UacFeedFiltersValidator(filterSchema)

                        property(PatchCampaignInternalRequest::feedFilters) {
                            checkBy(uacFeedFiltersValidator, When.notNull())
                        }
                    }

                    property(PatchCampaignInternalRequest::goals) {
                        check(CommonConstraints.notNull())
                        check(CollectionConstraints.minListSize(1))
                        check({ goals ->
                            val availableRequestedCounters = availableCounters.intersect(request.counters ?: setOf())

                            val counterIds = goals?.map { goalsByIds[it.goalId]?.counterId }
                            val counterId = counterIds?.firstOrNull()
                            if (counterIds?.any { it != counterId || !availableRequestedCounters.contains(it) } != false) {
                                invalidValue()
                            } else null
                        }, When.isValid())
                    }

                    // В случае Еком сценария недельный бюджет опционален, но если присутствует, то должен быть не меньше
                    // трёх (по числе создаваемых кампаний) минимальных сумм недельного бюджета для валюты клиента
                    val workCurrency = clientService.getWorkCurrency(clientId)
                    val minWeekLimit = workCurrency.minAutobudget.multiply(BigDecimal(3))
                    property(PatchCampaignInternalRequest::weekLimit) {
                        check(notLessThan(minWeekLimit), When.notNull())
                    }
                }

                property(request::relevanceMatch) {
                    check(
                        CommonConstraints.isNull(),
                        When.isTrue(uacYdbCampaign.advType != TEXT || !relevanceMatchCategoriesAreAllowed)
                    )
                    check({ relevanceMatch ->
                        // Если категории не переданы - все включены. Если переданы - EXACT должен быть всегда включен
                        if (relevanceMatch != null
                            && relevanceMatch.active
                            && relevanceMatch.categories.isNotEmpty()
                            && !relevanceMatch.categories.contains(UacRelevanceMatchCategory.EXACT_MARK)
                        ) {
                            invalidValue()
                        } else null
                    }, When.isValid())
                }

                property(request::showTitleAndBody) {
                    check(
                        CommonConstraints.notTrue(),
                        When.isFalse(
                            featureService.isEnabledForClientId(
                                client.clientId,
                                FeatureName.DISABLE_VIDEO_CREATIVE
                            )
                        )
                    )
                }

                property(request::altAppStores) {
                    check(CommonConstraints.isNull(), When.isFalse(altAppStoresAreAllowed))
                }
            }
        }

        fun apply(): Result<UacYdbCampaign> {
            val shouldUpdateAdsDeferred = !uacYdbCampaign.isDraft
            val shouldCheckAudienceSegmentsDeferred = featureService.isEnabledForClientId(
                clientId,
                FeatureName.CHECK_AUDIENCE_SEGMENTS_DEFERRED
            ) && audienceSegmentsService.hasProcessedSegments(
                request.retargetingCondition,
                client.login
            )
            val updatedUacYdbCampaign =
                uacYdbCampaign.updateModel(request, shouldUpdateAdsDeferred, shouldCheckAudienceSegmentsDeferred)

            val updateCampaignInDirectResult = updateCampaignInDirect(
                uacYdbCampaign = uacYdbCampaign,
                updatedUacYdbCampaign = updatedUacYdbCampaign,
                contents = contentById.values
            )

            if (!updateCampaignInDirectResult.isSuccessful) {
                return Result.broken(updateCampaignInDirectResult.validationResult)
            }

            if (
                uacYdbCampaign.advType == MOBILE_CONTENT
                && (request.adjustments != null || request.retargetingCondition != null)
            ) {
                uacAdjustmentsService.updateBidModifiers(
                    directCampaignId,
                    operator,
                    client,
                    request.adjustments,
                    request.retargetingCondition?.id,
                )
            }

            val fullyUpdatedUacYdbCampaign = updateBrief(
                updatedUacYdbCampaign,
                request,
                client.clientId,
                multipleAdsInUc
            )
            if (shouldUpdateAdsDeferred) {
                uacBannerService.updateAdsDeferred(clientId, operatorUid, updatedUacYdbCampaign.id)
            }
            if (shouldCheckAudienceSegmentsDeferred) {
                audienceSegmentsService.checkAudienceSegmentsDeferred(clientId, operatorUid, updatedUacYdbCampaign.id)
            }

            return Result.successful(fullyUpdatedUacYdbCampaign)
        }

        private fun UacYdbCampaign.updateModel(
            request: PatchCampaignInternalRequest,
            shouldUpdateAdsDeferred: Boolean,
            shouldCheckAudienceSegmentsDeferred: Boolean,
        ): UacYdbCampaign {
            return when (advType) {
                MOBILE_CONTENT -> updateMobileContentModel(
                    request,
                    shouldUpdateAdsDeferred,
                    shouldCheckAudienceSegmentsDeferred
                )

                else -> updateModelSimple(request, shouldUpdateAdsDeferred)
            }
        }

        private fun UacYdbCampaign.updateMobileContentModel(
            request: PatchCampaignInternalRequest,
            shouldUpdateAdsDeferred: Boolean,
            shouldCheckAudienceSegmentsDeferred: Boolean
        ): UacYdbCampaign {
            val updatedAppId = request.appId ?: appId
            val updatedAppInfo = uacMobileAppService.getAppInfo(updatedAppId)
            val updatedTrackingUrl = if (request.trackingUrl?.isBlank() == true) null
            else request.trackingUrl?.let { trackingUrl ->
                uacMobileAppService.getTrackingUrl(trackingUrl, updatedAppInfo)?.getUrl()
            } ?: trackingUrl
            val updatedImpressionUrl = if (request.impressionUrl?.isBlank() == true) null
            else request.impressionUrl?.let { impressionUrl ->
                uacMobileAppService.getImpressionUrl(impressionUrl, updatedAppInfo)?.getUrl()
            } ?: impressionUrl
            val uacAdjustments = if (request.adjustments != null) toUacAdjustments(request.adjustments)
            else adjustments

            return copy(
                name = request.displayName ?: name,
                cpa = request.cpa ?: cpa,
                crr = null,
                weekLimit = request.weekLimit ?: weekLimit,
                regions = request.regions ?: regions,
                minusRegions = request.minusRegions ?: minusRegions,
                storeUrl = request.href ?: storeUrl,
                appId = updatedAppId,
                targetId = request.targetId ?: targetId,
                trackingUrl = updatedTrackingUrl,
                impressionUrl = updatedImpressionUrl,
                targetStatus = request.targetStatus ?: targetStatus,
                adultContentEnabled = request.adultContentEnabled ?: adultContentEnabled,
                options = request.limitPeriod?.let { UacCampaignOptions(it) } ?: options,
                hyperGeoId = request.hyperGeoId ?: hyperGeoId,
                keywords = request.keywords ?: keywords,
                minusKeywords = request.minusKeywords ?: minusKeywords,
                socdem = request.socdem ?: socdem,
                deviceTypes = request.deviceTypes ?: deviceTypes,
                goals = request.goals ?: goals,
                counters = request.counters ?: counters,
                permalinkId = request.permalinkId ?: permalinkId,
                phoneId = request.phoneId ?: phoneId,
                timeTarget = request.timeTarget ?: timeTarget,
                strategy = request.strategy ?: if (request.cpa != null) null else strategy, // temporary until RMP-2996
                retargetingCondition = request.retargetingCondition,
                briefSynced = getUpdatedBriefSynced(briefSynced, shouldUpdateAdsDeferred),
                strategyPlatform = request.strategyPlatform ?: strategyPlatform,
                isEcom = false,
                feedId = null,
                feedFilters = null,
                trackingParams = null,
                uacBrandsafety = request.uacBrandsafety ?: uacBrandsafety,
                uacDisabledPlaces = request.uacDisabledPlaces ?: uacDisabledPlaces,
                adjustments = uacAdjustments,
                recommendationsManagementEnabled = request.isRecommendationsManagementEnabled
                    ?: recommendationsManagementEnabled,
                priceRecommendationsManagementEnabled = request.isPriceRecommendationsManagementEnabled
                    ?: priceRecommendationsManagementEnabled,
                relevanceMatch = request.relevanceMatch ?: relevanceMatch,
                showTitleAndBody = request.showTitleAndBody,
                audienceSegmentsSynchronized = getUpdatedaudienceSegmentsSynchronized(
                    audienceSegmentsSynchronized, shouldCheckAudienceSegmentsDeferred
                )
            )
        }

        private fun UacYdbCampaign.updateModelSimple(
            request: PatchCampaignInternalRequest,
            shouldUpdateAdsDeferred: Boolean,
        ): UacYdbCampaign {
            return copy(
                name = request.displayName ?: name,
                cpa = request.cpa,
                crr = request.crr,
                weekLimit = request.weekLimit ?: weekLimit,
                regions = request.regions,
                minusRegions = request.minusRegions,
                storeUrl = request.href ?: storeUrl,
                targetId = request.targetId,
                targetStatus = request.targetStatus ?: targetStatus,
                options = request.limitPeriod?.let { UacCampaignOptions(it) },
                hyperGeoId = request.hyperGeoId,
                keywords = request.keywords,
                minusKeywords = request.minusKeywords,
                socdem = request.socdem,
                deviceTypes = request.deviceTypes,
                inventoryTypes = request.inventoryTypes,
                goals = request.goals,
                counters = request.counters,
                permalinkId = request.permalinkId,
                phoneId = request.phoneId,
                calltrackingSettingsId = request.calltrackingSettingsId,
                timeTarget = request.timeTarget,
                strategy = request.strategy,
                retargetingCondition = request.retargetingCondition,
                videosAreNonSkippable = request.videosAreNonSkippable,
                zenPublisherId = request.zenPublisherId,
                brandSurveyId = request.brandSurveyId,
                briefSynced = getUpdatedBriefSynced(briefSynced, shouldUpdateAdsDeferred),
                showsFrequencyLimit = request.showsFrequencyLimit,
                isEcom = request.isEcom ?: isEcom,
                feedId = request.feedId ?: feedId,
                feedFilters = request.feedFilters ?: feedFilters,
                trackingParams = request.trackingParams,
                cpmAssets = request.cpmAssets,
                campaignMeasurers = request.campaignMeasurers,
                uacDisabledPlaces = request.uacDisabledPlaces,
                uacBrandsafety = request.uacBrandsafety,
                recommendationsManagementEnabled = request.isRecommendationsManagementEnabled
                    ?: recommendationsManagementEnabled,
                priceRecommendationsManagementEnabled = request.isPriceRecommendationsManagementEnabled
                    ?: priceRecommendationsManagementEnabled,
                relevanceMatch = request.relevanceMatch,
                showTitleAndBody = request.showTitleAndBody,
                bizLandingId = request.bizLandingId,
                searchLift = request.searchLift,
            )
        }

        private fun getUpdatedBriefSynced(
            oldBriefSynced: Boolean?,
            shouldUpdateAdsDeferred: Boolean
        ) = (oldBriefSynced ?: true) && !shouldUpdateAdsDeferred

        private fun getUpdatedaudienceSegmentsSynchronized(
            oldaudienceSegmentsSynchronized: Boolean?,
            shouldCheckAudienceSegmentsDeferred: Boolean
        ) = (oldaudienceSegmentsSynchronized ?: true) && !shouldCheckAudienceSegmentsDeferred

        private fun updateCampaignInDirect(
            uacYdbCampaign: UacYdbCampaign,
            updatedUacYdbCampaign: UacYdbCampaign,
            contents: Collection<Content>,
        ): Result<*> {
            return when (updatedUacYdbCampaign.advType) {
                MOBILE_CONTENT -> updateMobileContentCampaign(uacYdbCampaign, updatedUacYdbCampaign)
                TEXT -> updateTextCampaign(updatedUacYdbCampaign)
                CPM_BANNER -> updateCpmBannerCampaign(uacYdbCampaign, contents)
                else -> throw RuntimeException("Unsupported adv type: ${updatedUacYdbCampaign.advType}")
            }
        }

        private fun updateTextCampaign(updatedUacYdbCampaign: UacYdbCampaign): Result<Long> {
            val campaign = uacCampaignsCoreService.getCampaign(
                client.clientId,
                directCampaignId,
                TextCampaign::class
            )!!
            val currencyCode = clientService.getWorkCurrency(client.clientId).code
            val gdUpdateUcCampaign =
                toGdUpdateUcCampaignInput(updatedUacYdbCampaign, campaign, request, contentById.values, currencyCode)

            val updatingGroupVr =
                ucCampaignMutationService.validateUpdatingGroupForUac(gdUpdateUcCampaign, operatorUid, client.clientId)
            if (updatingGroupVr.hasAnyErrors()) {
                return Result.broken(updatingGroupVr)
            }
            val newMinusKeywords =
                if (updateMinusKeywords) updatedUacYdbCampaign.minusKeywords else campaign.minusKeywords

            return ucCampaignMutationService.updateCampaign(
                gdUpdateUcCampaign,
                newMinusKeywords,
                operatorUid,
                client
            )
        }

        private fun updateMobileContentCampaign(
            uacYdbCampaign: UacYdbCampaign,
            updatedUacYdbCampaign: UacYdbCampaign,
        ): Result<*> {
            val regions = updatedUacYdbCampaign.regions ?: uacYdbCampaign.regions
            val minusRegions = updatedUacYdbCampaign.minusRegions ?: uacYdbCampaign.minusRegions
            val geo = getGeoForUacGroups(regions, minusRegions)
            if (updatedUacYdbCampaign.keywords != null || geo.isNotEmpty()) {
                val adGroupVr = uacCampaignValidationService.validateUacAdGroup(
                    clientId, updatedUacYdbCampaign.keywords, uacYdbCampaign.keywords, geo
                )
                if (!adGroupVr.isSuccessful) {
                    val vrErrors: String = adGroupVr.validationResult?.flattenErrors()!!.joinToString("; ")
                    getLogger().error("error while validating groups: {}", vrErrors)
                    return adGroupVr
                }
            }
            if (uacYdbCampaign.appId != updatedUacYdbCampaign.appId) {
                return recreateCampaign(updatedUacYdbCampaign, directCampaignId)
            } else {
                val campaign =
                    rmpCampaignService.getMobileContentCampaign(clientId, directCampaignId)!!

                val oldUpdateCampaignRequest =
                    uacYdbCampaign.toUpdateMobileContentCampaignRequest(campaign.alternativeAppStores)
                val newAltAppStores = if (this.request.altAppStores == null) {
                    campaign.alternativeAppStores
                } else if (this.request.altAppStores.isEmpty()) {
                    EnumSet.noneOf(MobileAppAlternativeStore::class.java)
                } else {
                    EnumSet.copyOf(this.request.altAppStores.mapToSet { it.toCoreType() })
                }
                val newUpdateCampaignRequest =
                    updatedUacYdbCampaign.toUpdateMobileContentCampaignRequest(newAltAppStores)

                if (oldUpdateCampaignRequest != newUpdateCampaignRequest) {
                    val client = clientService.getClient(clientId)!!
                    val updateResult =
                        rmpCampaignService.updateRmpCampaign(
                            client,
                            operator,
                            campaign,
                            newUpdateCampaignRequest,
                        )

                    if (!updateResult.isSuccessful) {
                        return updateResult
                    }
                }

                return Result.successful(null)
            }
        }

        private fun updateCpmBannerCampaign(
            uacYdbCampaign: UacYdbCampaign,
            contents: Collection<Content>,
        ): Result<Long> {
            val updateDataContainer = uacModifyCampaignDataContainerFactory.dataContainerFromPatchRequest(
                uacYdbCampaign.id,
                uacYdbCampaign.advType,
                uacYdbCampaign.createdAt,
                uacYdbCampaign.startedAt,
                this.request,
                FeatureName.ADVANCED_GEOTARGETING.enabled()
            )
            cpmBannerService.fillCampaign(clientId, updateDataContainer)

            val cpmBannerCampaign = UacCampaignConverter.toCpmBannerCampaign(
                updateDataContainer,
                this.operator,
                sspPlatformsRepository,
                hostingsHandler
            )
            cpmBannerCampaign.id = directCampaignId

            val addingGroupVr: ValidationResult<AdGroup, Defect<*>> =
                uacCampaignValidationService.validateAddingCpmGroup(updateDataContainer, contents, clientId)

            if (addingGroupVr.hasAnyErrors()) {
                return Result.broken(addingGroupVr)
            }

            val result = cpmBannerCampaignService.updateCpmBannerCampaign(
                operator,
                UidAndClientId.of(operatorUid, clientId),
                cpmBannerCampaign,
                toCoreCpmCampaignModelChanges(cpmBannerCampaign)
            )

            if (!result.isSuccessful) {
                if (result.validationResult.subResults != null && result.validationResult.subResults.isNotEmpty()) {
                    return Result.broken(result.validationResult.subResults.values.first())
                }

                return Result.broken(result.validationResult)
            }

            return Result.successful(cpmBannerCampaign.id)
        }

        private fun recreateCampaign(
            updatedUacYdbCampaign: UacYdbCampaign,
            directCampaignId: Long,
        ): Result<*> {
            return recreateCampaignInDirect(updatedUacYdbCampaign, directCampaignId, operator, clientId, operatorUid)
        }

        private fun UacYdbCampaign.toUpdateMobileContentCampaignRequest(
            alternativeAppStores: EnumSet<MobileAppAlternativeStore>? = null
        ): UpdateUacCampaignRequest {
            return UpdateUacCampaignRequest(
                name = name,
                trackingUrl = trackingUrl,
                impressionUrl = impressionUrl,
                cpa = cpa?.let { moneyToDb(it) },
                weekLimit = weekLimit?.let { moneyToDb(it) },
                targetId = targetId,
                adultContentEnabled = adultContentEnabled,
                strategyPlatform = strategyPlatform,
                strategy = strategy,
                uacDisabledPlaces = uacDisabledPlaces,
                alternativeAppStores = alternativeAppStores,
            )
        }
    }

    abstract fun updateBrief(
        updatedUacYdbCampaign: UacYdbCampaign,
        request: PatchCampaignInternalRequest,
        clientId: ClientId,
        multipleAdsInUc: Boolean,
    ): UacYdbCampaign

    abstract fun setDirectCampaignStatusToCreated(
        ydbCampaign: UacYdbCampaign,
        operator: User,
        subjectUser: User,
        changes: KtModelChanges<String, UacYdbCampaign>,
    ): Boolean

    abstract fun recreateCampaignInDirect(
        updatedUacYdbCampaign: UacYdbCampaign,
        directCampaignId: Long,
        operator: User,
        clientId: ClientId,
        operatorUid: Long,
    ): Result<*>

    fun getCampaignStatusShow(directCampaignId: Long): Boolean? {
        val shard = shardHelper.getShardByCampaignId(directCampaignId)
        return baseUacCampaignService.getCampaignStatusShow(shard, directCampaignId)
    }
}
