package ru.yandex.direct.web.entity.uac.validation

import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_DOMAIN_LENGTH
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.uniqueStrings
import ru.yandex.direct.core.entity.campaign.service.validation.DisableDomainValidationService
import ru.yandex.direct.core.entity.client.service.ClientLimitsService
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacDisabledPlaces
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler
import ru.yandex.direct.utils.FunctionalUtils
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.validation.wrapper.DefaultValidator

class UacDisabledPlacesValidator (
    private val clientLimitsService: ClientLimitsService,
    private val clientId: ClientId,
    private val hostingsHandler: HostingsHandler,
    private val disableDomainValidationService: DisableDomainValidationService,
    private val sspPlatformsRepository: SspPlatformsRepository,

    ): DefaultValidator<UacDisabledPlaces> {

    override fun apply(t: UacDisabledPlaces): ValidationResult<UacDisabledPlaces, Defect<*>> {
        val videoBlacklistSizeLimit: Int = clientLimitsService
            .getClientLimits(clientId)
            .videoBlacklistSizeLimitOrDefault.toInt()
        val generalBlacklistSizeLimit = clientLimitsService
            .getClientLimits(clientId)
            .generalBlacklistSizeLimitOrDefault.toInt()
        val knownSsp: List<String> = sspPlatformsRepository.allSspPlatforms
        return validateObject(t) {
            val disabledSsps = t.disabledPlaces?.map { t: String -> hostingsHandler.stripWww(t) }
                ?.flatMap { t: String -> FunctionalUtils.filterList(knownSsp) { another: String -> t.equals(another, ignoreCase = true) } }
                .orEmpty()

            listProperty(UacDisabledPlaces::disabledPlaces) {
                check(maxListSize(generalBlacklistSizeLimit))
                check { t ->
                    uniqueStrings().apply(t?.map { hostingsHandler.stripWww(it)})
                }
                checkEach(maxStringLength(MAX_DOMAIN_LENGTH))
                checkEach { t: String ->
                    val stripped = hostingsHandler.stripWww(t)
                    if (disabledSsps.contains(t)) {
                        null
                    } else {
                        CampaignConstraints.isValidDisabledDomain(clientId, disableDomainValidationService)
                            .apply(stripped)
                    }
                }
            }

            listProperty(UacDisabledPlaces::disabledVideoAdsPlaces) {
                check(maxListSize(videoBlacklistSizeLimit))
                check { t ->
                    uniqueStrings().apply(t?.map { hostingsHandler.stripWww(it)})
                }
                checkEach(maxStringLength(MAX_DOMAIN_LENGTH))
                checkEach(Constraint { t ->
                    CampaignConstraints.isValidDisabledDomain(clientId, disableDomainValidationService)
                        .apply(hostingsHandler.stripWww(t))
                })
            }

            listProperty(UacDisabledPlaces::disallowedPageIds) {
                check(maxListSize(CampaignConstants.MAX_ALLOWED_PAGE_IDS_COUNT))
                checkEach(CommonConstraints.validId())
            }

        }
    }
}
