package ru.yandex.direct.web.entity.uac.validation

import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterTab
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema
import ru.yandex.direct.core.entity.performancefilter.utils.PerformanceFilterUtils
import ru.yandex.direct.core.entity.performancefilter.validation.FilterConditionsValidator
import ru.yandex.direct.core.entity.uac.converter.UacEcomConverter.toSmartConditions
import ru.yandex.direct.core.entity.uac.model.UacFeedFilter
import ru.yandex.direct.core.entity.uac.model.UacFeedFilterCondition
import ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.validation.wrapper.DefaultValidator

class UacFeedFiltersValidator(
    private val filterSchema: FilterSchema
) : DefaultValidator<List<UacFeedFilter>> {

    override fun apply(feedFilters: List<UacFeedFilter>): ValidationResult<List<UacFeedFilter>, Defect<*>> {
        val uacToPerfConditions = feedFilters.associate {
            it.conditions to it.conditions.toSmartConditions(filterSchema)
        }

        return validateList(feedFilters) {
            checkEachBy { filter -> hasDuplicate(filter, feedFilters, uacToPerfConditions) }
            checkEachBy { filter -> validateFilter(filter, uacToPerfConditions) }
        }
    }

    private fun hasDuplicate(
        filter: UacFeedFilter,
        filters: List<UacFeedFilter>,
        uacToPerfConditions: Map<List<UacFeedFilterCondition>, List<PerformanceFilterCondition<Any>>>
    ): ValidationResult<UacFeedFilter, Defect<*>> {
        val conditions = uacToPerfConditions[filter.conditions]
        val otherFilters = filters.filter { it !== filter }

        val isDuplicate = otherFilters.any { PerformanceFilterUtils.isEqual(uacToPerfConditions[it.conditions], conditions) }

        return if (isDuplicate) {
            ValidationResult.failed(filter, duplicatedObject())
        } else {
            ValidationResult.success(filter)
        }
    }

    private fun validateFilter(
        filter: UacFeedFilter,
        uacToPerfConditions: Map<List<UacFeedFilterCondition>, List<PerformanceFilterCondition<Any>>>
    ): ValidationResult<UacFeedFilter, Defect<*>> {
        return validateObject(filter) {
            listProperty(UacFeedFilter::conditions) {
                checkBy { conditions -> validateConditions(conditions, uacToPerfConditions) }
            }
        }
    }

    private fun validateConditions(
        conditions: List<UacFeedFilterCondition>,
        uacToPerfConditions: Map<List<UacFeedFilterCondition>, List<PerformanceFilterCondition<Any>>>
    ): ValidationResult<List<UacFeedFilterCondition>, Defect<*>> {
        val res = FilterConditionsValidator(filterSchema, FeedFilterTab.ALL_PRODUCTS)
            .apply(uacToPerfConditions[conditions])

        return if (res.hasAnyErrors()) {
            ValidationResult(conditions, res.errors, res.warnings, res.subResults)
        } else {
            ValidationResult.success(conditions)
        }
    }
}
