package ru.yandex.direct.web.entity.uac.validation

import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoal
import ru.yandex.direct.core.entity.uac.model.UacRetargetingConditionRuleGoalType
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.direct.validation.wrapper.DefaultValidator

class UacRetargetingConditionRuleGoalValidator(
    private val existingGoalIds: Map<Long, String>,
    private val availableAudienceGoalIds: Set<Long>,
    private val existingMetrikaSegmentIds: Set<Long>,
) : DefaultValidator<UacRetargetingConditionRuleGoal> {
    companion object {
        val MOBILE_TYPES = setOf(UacRetargetingConditionRuleGoalType.LAL, UacRetargetingConditionRuleGoalType.MOBILE)
        val AUDIENCE_TYPES =
            setOf(UacRetargetingConditionRuleGoalType.AUDIENCE, UacRetargetingConditionRuleGoalType.LAL_AUDIENCE)
        val METRIKA_TYPES =
            setOf(UacRetargetingConditionRuleGoalType.SEGMENT, UacRetargetingConditionRuleGoalType.LAL_SEGMENT)
    }

    override fun apply(goal: UacRetargetingConditionRuleGoal): ValidationResult<UacRetargetingConditionRuleGoal, Defect<*>> {
        return validateObject(goal) {
            property(UacRetargetingConditionRuleGoal::time) {
                check(notNull(), When.isTrue(goal.type == UacRetargetingConditionRuleGoalType.MOBILE))
            }
            property(UacRetargetingConditionRuleGoal::type) {
                check(notNull())
            }
            property(UacRetargetingConditionRuleGoal::id) {
                check(
                    Constraint.fromPredicate(
                        { existingGoalIds.contains(goal.id) },
                        CommonDefects.objectNotFound()
                    ), When.isTrue(goal.type in MOBILE_TYPES)
                )
                check(
                    Constraint.fromPredicate(
                        { availableAudienceGoalIds.contains(goal.id) },
                        CommonDefects.objectNotFound()
                    ), When.isTrue(goal.type in AUDIENCE_TYPES)
                )
                check(
                    Constraint.fromPredicate(
                        { existingMetrikaSegmentIds.contains(goal.id) },
                        CommonDefects.objectNotFound()
                    ), When.isTrue(goal.type in METRIKA_TYPES)
                )
            }
        }
    }
}
