package ru.yandex.direct.web.entity.user.controller

import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.ResponseBody
import org.springframework.web.bind.annotation.RestController
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.user.service.BlackboxUserService
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.i18n.Language
import ru.yandex.direct.web.core.model.WebErrorResponse
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha
import ru.yandex.direct.web.core.security.configuration.BlackboxWebAuthenticationConfiguration.PUBLIC_BASE_URL
import ru.yandex.direct.web.entity.user.model.PublicFeature
import ru.yandex.direct.web.entity.user.model.PublicOperatorInfo
import ru.yandex.direct.web.entity.user.model.toResponse

@Api(tags = ["public-user"])
@RestController
@RequestMapping(PUBLIC_BASE_URL + "user/", produces = [MediaType.APPLICATION_JSON_VALUE])
class PublicUserController(
    private val directWebAuthenticationSource: DirectWebAuthenticationSource,
    private val blackboxUserService: BlackboxUserService,
    private val featureService: FeatureService,
    private val userService: UserService,
) {
    @DisableAutoCaptcha
    @ApiOperation(value = "getLogin", httpMethod = "GET", nickname = "getLogin")
    @ApiResponses(
        ApiResponse(code = 200, message = "OK"),
        ApiResponse(code = 404, message = "Not found", response = WebErrorResponse::class)
    )
    @GetMapping(path = ["/operator_info"])
    @ResponseBody
    fun getLogin(): ResponseEntity<Any> {
        val operator = directWebAuthenticationSource.authentication.operator
        if (operator == null || operator.login == null) {
            return ResponseEntity.notFound().build()
        }
        val exists = userService.loginExists(operator.login)
        val user = blackboxUserService.getUserInfo(operator.uid)
        val avatarId = user?.avatarId ?: "0/0"

        val publicOperatorInfo = PublicOperatorInfo(operator.login, avatarId, exists, operator.role,
            operator.statusBlocked ?: false, user?.lang ?: Language.RU).toResponse()

        return ResponseEntity.ok(publicOperatorInfo)
    }

    @DisableAutoCaptcha
    @ApiOperation(value = "getFeatures", httpMethod = "GET", nickname = "getFeatures")
    @ApiResponses(
        ApiResponse(code = 200, message = "OK"),
        ApiResponse(code = 404, message = "Not found", response = WebErrorResponse::class)
    )
    @GetMapping("/features")
    @ResponseBody
    fun getFeature(): ResponseEntity<Any> {
        val operator = directWebAuthenticationSource.authentication.operator
        if (operator == null || operator.login == null) {
            return ResponseEntity.notFound().build()
        }
        val enabled = featureService.isEnabledForUid(operator.uid, listOf(FeatureName.USE_KOTLIN_FOR_V1_USER))
        val response = PublicFeature(enabled).toResponse()
        return ResponseEntity.ok(response)
    }
}
