package ru.yandex.direct.web.entity.useractionlog;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.dataloader.DataLoader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;

@Component
@ParametersAreNonnullByDefault
// Обязательно должен иметь какой-нибудь Scope, не связанный с постоянно живущими синглтонами. Внутри DataLoader'ов
// встроен топорный вечный невытесняющий кеш, который, впрочем, полезен в пределах одного HTTP-запроса.
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
public class ClientNameByClientIdDataLoader extends DataLoader<Long, String> {
    @Autowired
    public ClientNameByClientIdDataLoader(ShardHelper shardHelper, ClientRepository clientRepository) {
        super(clientIds -> {
            ShardedData<Long> clientIdsByShards = shardHelper.groupByShard(
                    clientIds.stream().filter(Objects::nonNull).collect(Collectors.toSet()),
                    ShardKey.CLIENT_ID);
            Map<Long, String> nameById = clientIdsByShards.stream()
                    .mapValues(ids -> ids.stream().map(ClientId::fromLong).collect(Collectors.toSet()))
                    .mapKeyValue(clientRepository::get)
                    .flatMap(Collection::stream)
                    .mapToEntry(Client::getName)
                    .mapKeys(Client::getId)
                    .toMap();
            return CompletableFuture.completedFuture(clientIds.stream()
                    .map(id -> nameById.getOrDefault(id, ""))
                    .collect(Collectors.toList()));
        });
    }
}
