package ru.yandex.direct.web.entity.useractionlog;

import java.util.Collection;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.dataloader.DataLoader;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

/**
 * DataLoader, который получает UserView на основе uid.
 */
@Component
@ParametersAreNonnullByDefault
// Обязательно должен иметь какой-нибудь Scope, не связанный с постоянно живущими синглтонами. Внутри DataLoader'ов
// встроен топорный вечный невытесняющий кеш, который, впрочем, полезен в пределах одного HTTP-запроса.
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
public class UserViewByUidDataLoader extends DataLoader<Long, UserView> {
    @Autowired
    public UserViewByUidDataLoader(ShardHelper shardHelper, UserRepository userRepository) {
        super(uids -> {
            Map<Long, UserView> usersMap = shardHelper
                    .groupByShard(
                            StreamEx.of(uids.iterator())
                                    .filter(Objects::nonNull)
                                    .map(Long.class::cast)
                                    .collect(Collectors.toList()),
                            ShardKey.UID)
                    .stream()
                    .mapKeyValue(userRepository::fetchByUids)
                    .flatMap(Collection::stream)
                    .collect(Collectors.toMap(User::getUid, u -> new UserView(u.getUid(), u.getLogin())));
            return CompletableFuture.completedFuture(uids.stream()
                    .map(i -> usersMap.getOrDefault(i, new UserView(i, "")))
                    .collect(Collectors.toList()));
        });
    }
}
