package ru.yandex.direct.web.entity.vcard.service;

import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.vcard.container.AssignVcardRequest;
import ru.yandex.direct.core.entity.vcard.container.SaveVcardRequest;
import ru.yandex.direct.core.entity.vcard.container.UnassignVcardRequest;
import ru.yandex.direct.core.entity.vcard.model.InstantMessenger;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.PointPrecision;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.service.ManageVcardService;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.vcard.model.AssignVcardResponse;
import ru.yandex.direct.web.entity.vcard.model.AssignVcardResult;
import ru.yandex.direct.web.entity.vcard.model.AssignVcardWebRequest;
import ru.yandex.direct.web.entity.vcard.model.DeprecatedWebVcard;
import ru.yandex.direct.web.entity.vcard.model.SaveVcardResponse;
import ru.yandex.direct.web.entity.vcard.model.SaveVcardResult;
import ru.yandex.direct.web.entity.vcard.model.SaveVcardWebRequest;
import ru.yandex.direct.web.entity.vcard.model.UnassignVcardWebRequest;
import ru.yandex.direct.web.entity.vcard.model.WebInstantMessenger;
import ru.yandex.direct.web.entity.vcard.model.WebPhone;
import ru.yandex.direct.web.entity.vcard.model.WebPointOnMap;
import ru.yandex.direct.web.entity.vcard.model.WebVcard;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.common.util.TextUtils.smartStrip;
import static ru.yandex.direct.core.entity.vcard.VcardWorktimeUtils.toWebWorktimes;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.web.entity.vcard.presentations.ManageVcardsPathConverters.ASSIGN_VCARD_PATH_CONVERTER;
import static ru.yandex.direct.web.entity.vcard.presentations.ManageVcardsPathConverters.SAVE_VCARD_PATH_CONVERTER;
import static ru.yandex.direct.web.entity.vcard.presentations.ManageVcardsPathConverters.UNASSIGN_VCARD_PATH_CONVERTER;
import static ru.yandex.direct.web.entity.vcard.presentations.VcardPathConverter.INSTANT_MESSENGER_PATH_CONVERTER;
import static ru.yandex.direct.web.entity.vcard.presentations.VcardPathConverter.PHONE_PATH_CONVERTER;
import static ru.yandex.direct.web.entity.vcard.presentations.VcardPathConverter.POINT_ON_MAP_PATH_CONVERTER;
import static ru.yandex.direct.web.entity.vcard.presentations.VcardPathConverter.VCARD_PATH_CONVERTER;

@Service
@ParametersAreNonnullByDefault
public class WebVcardsService {
    private static final Logger logger = LoggerFactory.getLogger(WebVcardsService.class);

    private final DirectWebAuthenticationSource authenticationSource;
    private final ValidationResultConversionService validationResultConversionService;
    private final ManageVcardService manageVcardService;
    private final PathNodeConverterProvider pathNodeConverterProvider;


    @Autowired
    public WebVcardsService(
            DirectWebAuthenticationSource authenticationSource,
            ValidationResultConversionService validationResultConversionService,
            ManageVcardService manageVcardService,
            PathNodeConverterProvider defaultWebPathNodeConverterProvider) {
        this.authenticationSource = authenticationSource;
        this.validationResultConversionService = validationResultConversionService;
        this.manageVcardService = manageVcardService;

        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(Vcard.class, VCARD_PATH_CONVERTER)
                .register(Phone.class, PHONE_PATH_CONVERTER)
                .register(InstantMessenger.class, INSTANT_MESSENGER_PATH_CONVERTER)
                .register(PointOnMap.class, POINT_ON_MAP_PATH_CONVERTER)
                .register(SaveVcardRequest.class, SAVE_VCARD_PATH_CONVERTER)
                .register(AssignVcardRequest.class, ASSIGN_VCARD_PATH_CONVERTER)
                .register(UnassignVcardRequest.class, UNASSIGN_VCARD_PATH_CONVERTER)
                .fallbackTo(defaultWebPathNodeConverterProvider)
                .build();
    }

    public WebResponse saveVcard(SaveVcardWebRequest webRequest) {
        DirectAuthentication authentication = authenticationSource.getAuthentication();
        ClientId clientId = authentication.getSubjectUser().getClientId();
        Long operatorUid = authentication.getOperator().getUid();
        logger.debug("uid: {}, clientId: {}", authentication.getSubjectUser().getUid(), clientId);

        SaveVcardRequest modelRequest = convertSaveVcardWebRequestToModel(webRequest);

        Result<Long> operationResult = manageVcardService.saveVcard(
                operatorUid, clientId, modelRequest);

        if (hasValidationIssues(operationResult)) {
            return validationResultConversionService.buildValidationResponse(
                    operationResult.getValidationResult(), pathNodeConverterProvider);
        }
        Long newVcardId = operationResult.getResult();
        Vcard newVcard = manageVcardService.getVCardById(operatorUid, clientId, newVcardId);
        // если визитка редактируется во всех объявлениях, то нужно вернуть id старой визитки
        Long oldVcardId = modelRequest.getBannerIds() == null ? modelRequest.getVcardId() : null;

        SaveVcardResult result = new SaveVcardResult()
                .withNewVcard(toDeprecatedWebVcard(newVcard))
                .withOldVcardId(oldVcardId);
        return new SaveVcardResponse().withResult(result);
    }

    public WebResponse assignVcard(AssignVcardWebRequest webRequest) {
        DirectAuthentication authentication = authenticationSource.getAuthentication();
        ClientId clientId = authentication.getSubjectUser().getClientId();
        Long operatorUid = authentication.getOperator().getUid();
        logger.debug("uid: {}, clientId: {}", authentication.getSubjectUser().getUid(), clientId);

        AssignVcardRequest modelRequest = convertAssignVcardWebRequestToModel(webRequest);

        Result<Long> operationResult = manageVcardService.assignVcard(
                operatorUid, clientId, modelRequest);

        if (hasValidationIssues(operationResult)) {
            return validationResultConversionService.buildValidationResponse(
                    operationResult.getValidationResult(), pathNodeConverterProvider);
        }
        Long newVcardId = operationResult.getResult();
        Vcard newVcard = manageVcardService.getVCardById(operatorUid, clientId, newVcardId);

        return new AssignVcardResponse().withResult(
                new AssignVcardResult().withNewVcard(toDeprecatedWebVcard(newVcard)));
    }

    public WebResponse unassignVcard(UnassignVcardWebRequest webRequest) {
        DirectAuthentication authentication = authenticationSource.getAuthentication();
        ClientId clientId = authentication.getSubjectUser().getClientId();
        logger.debug("uid: {}, clientId: {}", authentication.getSubjectUser().getUid(), clientId);

        UnassignVcardRequest modelRequest = convertUnassignVcardWebRequestToModel(webRequest);

        Result<Long> operationResult = manageVcardService.unassignVcard(
                authentication.getOperator().getUid(), clientId, modelRequest);

        if (hasValidationIssues(operationResult)) {
            return validationResultConversionService.buildValidationResponse(
                    operationResult.getValidationResult(), pathNodeConverterProvider);
        }
        return new WebSuccessResponse();
    }

    private SaveVcardRequest convertSaveVcardWebRequestToModel(SaveVcardWebRequest request) {
        return new SaveVcardRequest()
                .withCampaignId(request.getCid())
                .withBannerIds(request.getBids())
                .withVcardId(request.getVcardId())
                .withVcard(toVcard(request.getVcard()));
    }

    private AssignVcardRequest convertAssignVcardWebRequestToModel(AssignVcardWebRequest request) {
        return new AssignVcardRequest()
                .withCampaignId(request.getCid())
                .withBannerIds(request.getBids())
                .withVcardId(request.getVcardId());
    }

    private UnassignVcardRequest convertUnassignVcardWebRequestToModel(UnassignVcardWebRequest request) {
        return new UnassignVcardRequest()
                .withCampaignId(request.getCid())
                .withBannerIds(request.getBids());
    }

    /* из AddVCardsDelegate
        для полей визитки из запроса вызывается smartStrip
        не все поля в Vcard нужны для создания новой визитки (например lastChange)
        */
    @Nullable
    private static Vcard toVcard(@Nullable WebVcard webVcard) {
        return Optional.ofNullable(webVcard)
                .map(vcard -> new Vcard()
                        .withCampaignId(vcard.getCampaignId())
                        .withCountry(smartStripAndReplaceEmpty(vcard.getCountry()))
                        .withCity(smartStripAndReplaceEmpty(vcard.getCity()))
                        .withCompanyName(smartStripAndReplaceEmpty(vcard.getCompanyName()))
                        .withWorkTime(smartStripAndReplaceEmpty(vcard.getWorkTime()))
                        .withPhone(toVcardPhone(vcard.getPhone()))
                        .withStreet(smartStripAndReplaceEmpty(vcard.getStreet()))
                        .withHouse(smartStripAndReplaceEmpty(vcard.getHouse()))
                        .withBuild(smartStripAndReplaceEmpty(vcard.getBuild()))
                        .withApart(smartStripAndReplaceEmpty(vcard.getApart()))
                        .withInstantMessenger(toVcardInstantMessenger(vcard.getInstantMessenger()))
                        .withExtraMessage(smartStripAndReplaceEmpty(vcard.getExtraMessage()))
                        .withEmail(smartStripAndReplaceEmpty(vcard.getEmail()))
                        .withOgrn(smartStripAndReplaceEmpty(vcard.getOgrn()))
                        .withMetroId(vcard.getMetroId())
                        .withManualPoint(toVcardPointOnMap(vcard.getManualPoint()))
                        .withContactPerson(smartStripAndReplaceEmpty(vcard.getContactPerson())))
                .orElse(null);
    }

    @Nullable
    private static Phone toVcardPhone(@Nullable WebPhone phone) {
        return Optional.ofNullable(phone)
                .map(p -> new Phone()
                        .withCountryCode(smartStripAndReplaceEmpty(p.getCountryCode()))
                        .withCityCode(smartStripAndReplaceEmpty(p.getCityCode()))
                        .withPhoneNumber(smartStripAndReplaceEmpty(p.getPhoneNumber()))
                        .withExtension(smartStripAndReplaceEmpty(p.getExtension())))
                .orElse(null);
    }

    @Nullable
    private static InstantMessenger toVcardInstantMessenger(@Nullable WebInstantMessenger messenger) {
        if (messenger != null) {
            String imType = smartStripAndReplaceEmpty(messenger.getType());
            String imLogin = smartStripAndReplaceEmpty(messenger.getLogin());
            if (imType != null || imLogin != null) {
                return new InstantMessenger().withType(imType).withLogin(imLogin);
            }
        }
        return null;
    }

    @Nullable
    private static PointOnMap toVcardPointOnMap(@Nullable WebPointOnMap pointOnMap) {
        return Optional.ofNullable(pointOnMap)
                .map(p -> new PointOnMap().withX(p.getX())
                        .withY(p.getY())
                        .withX1(p.getX1())
                        .withY1(p.getY1())
                        .withX2(p.getX2())
                        .withY2(p.getY2()))
                .orElse(null);
    }

    @Nullable
    private static String smartStripAndReplaceEmpty(@Nullable String text) {
        String processedText = smartStrip(text);
        // Пустые строки приравниваем к null
        return StringUtils.defaultIfEmpty(processedText, null);
    }

    @Nullable
    private static DeprecatedWebVcard toDeprecatedWebVcard(@Nullable Vcard vcard) {
        if (vcard == null) {
            return null;
        }
        DeprecatedWebVcard webVcard = new DeprecatedWebVcard()
                .withId(vcard.getId())
                .withCampaignId(vcard.getCampaignId().toString())
                .withUid(vcard.getUid().toString())
                .withGeoId(vcard.getGeoId())
                .withCountry(vcard.getCountry())
                .withCity(vcard.getCity())
                .withStreet(vcard.getStreet())
                .withHouse(vcard.getHouse())
                .withBuild(vcard.getBuild())
                .withApart(vcard.getApart())
                .withMetroId(vcard.getMetroId())
                .withMetroName(vcard.getMetroName())
                .withCountryGeoId(vcard.getCountryGeoId())
                .withCompanyName(vcard.getCompanyName())
                .withContactPerson((vcard.getContactPerson()))
                .withEmail(vcard.getEmail())
                .withWorkTime(vcard.getWorkTime())
                .withWorkTimeList(toWebWorktimes(vcard.getWorkTime()))
                .withExtraMessage(vcard.getExtraMessage())
                .withOrgDetailsId(vcard.getOrgDetailsId())
                .withOgrn(vcard.getOgrn());

        addDeprecatedWebVcardPhone(webVcard, vcard.getPhone());
        addDeprecatedWebVcardInstantMessenger(webVcard, vcard.getInstantMessenger());
        addDeprecatedWebVcardPoints(webVcard, vcard.getManualPoint(), vcard.getAutoPoint());
        addDeprecatedWebVcardPrecision(webVcard, vcard.getPrecision());

        return webVcard;
    }

    private static void addDeprecatedWebVcardPhone(DeprecatedWebVcard webVcard, @Nullable Phone phone) {
        if (phone != null) {
            webVcard.withCountryCode(phone.getCountryCode())
                    .withCityCode(phone.getCityCode())
                    .withPhoneNumber(phone.getPhoneNumber())
                    .withExtension(defaultIfNull(phone.getExtension(), ""));
        }
    }

    private static void addDeprecatedWebVcardInstantMessenger(DeprecatedWebVcard webVcard,
                                                              @Nullable InstantMessenger messenger) {
        if (messenger != null) {
            webVcard.withInstantMessengerType(messenger.getType())
                    .withInstantMessengerLogin(messenger.getLogin());
        }
    }

    private static void addDeprecatedWebVcardPoints(DeprecatedWebVcard webVcard,
                                                    @Nullable PointOnMap manualPoint, @Nullable PointOnMap autoPoint) {
        webVcard.withManualPoint(getPointAsString(manualPoint))
                .withManualBounds(getBoundsAsString(manualPoint))
                .withAutoPoint(getPointAsString(autoPoint))
                .withAutoBounds(getBoundsAsString(autoPoint));
    }

    private static String getPointAsString(@Nullable PointOnMap point) {
        return (point == null) ? ""
                : StreamEx.of(point.getX(), point.getY()).nonNull().joining(",");
    }

    private static String getBoundsAsString(@Nullable PointOnMap point) {
        return (point == null) ? ""
                : StreamEx.of(point.getX1(), point.getY1(), point.getX2(), point.getY2()).nonNull().joining(",");
    }

    private static void addDeprecatedWebVcardPrecision(DeprecatedWebVcard webVcard,
                                                       @Nullable PointPrecision precision) {
        if (precision != null) {
            webVcard.withPrecision(StringUtils.lowerCase(precision.name()));
        }
    }
}
