package ru.yandex.direct.web.validation.kernel;

import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.presentation.DefectPresentationRegistry;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.MassResponse;
import ru.yandex.direct.web.validation.model.ValidationResponse;
import ru.yandex.direct.web.validation.model.WebDefect;
import ru.yandex.direct.web.validation.model.WebValidationResult;

import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Преобразование {@link ValidationResult} и {@link MassResult} в структуру web-api для фронта
 */
@Service
@ParametersAreNonnullByDefault
public class ValidationResultConversionService {
    private static final Path EMPTY_PATH = path();

    private final PathNodeConverterProvider defaultPathNodeConverterProvider;
    private final DefectPresentationRegistry<TranslatableWebDefect> defectPresentationRegistry;
    private final TranslationService translationService;

    @Autowired
    public ValidationResultConversionService(PathNodeConverterProvider defaultPathNodeConverterProvider,
                                             DefectPresentationRegistry<TranslatableWebDefect> defectPresentationRegistry,
                                             TranslationService translationService) {
        this.defaultPathNodeConverterProvider = defaultPathNodeConverterProvider;
        this.defectPresentationRegistry = defectPresentationRegistry;
        this.translationService = translationService;
    }

    /**
     * получить ответ для 1го элемента. Пригодится в случае, когда ручка работает с 1м элементом, а из ядра приходит
     * {@link MassResult}
     */
    public ValidationResponse buildSingleValidationResponse(MassResult<?> massResult) {
        @SuppressWarnings("unchecked")
        Result singleResult = massResult.getResult()
                .stream()
                .findFirst()
                .orElse((Result) massResult);
        return buildValidationResponse(singleResult);
    }

    public <T> ValidationResponse buildValidationResponse(Result<T> result) {
        return buildValidationResponse(result.getValidationResult());
    }

    public ValidationResponse buildValidationResponse(
            @Nullable ValidationResult<?, Defect> validationResult) {
        return new ValidationResponse(buildWebValidationResult(validationResult));
    }

    public ValidationResponse buildValidationResponse(
            @Nullable ValidationResult<?, Defect> validationResult,
            PathNodeConverterProvider pathNodeConverterProvider) {
        return new ValidationResponse(buildWebValidationResult(validationResult, EMPTY_PATH,
                pathNodeConverterProvider));
    }

    public <R extends MassResponse, A> R buildMassResponse(
            @Nullable ValidationResult<?, Defect> validationResult,
            A webResult,
            Supplier<R> resultSupplier) {
        WebValidationResult vr = buildWebValidationResult(validationResult);
        R r = resultSupplier.get();
        r.setResult(webResult);
        r.setWebValidationResult(vr);
        return r;
    }

    public WebValidationResult buildWebValidationResult(@Nullable ValidationResult<?, Defect> vr) {
        return buildWebValidationResult(vr, path(), defaultPathNodeConverterProvider);
    }

    public WebValidationResult buildWebValidationResult(@Nullable ValidationResult<?, Defect> vr,
                                                        Path path) {
        return buildWebValidationResult(vr, path, defaultPathNodeConverterProvider);
    }

    public WebValidationResult buildWebValidationResult(@Nullable ValidationResult<?, Defect> vr,
                                                        Path prefixPath, PathNodeConverterProvider pathNodeConverterProvider) {
        WebValidationResult result = new WebValidationResult();
        if (vr == null) {
            return result;
        }

        List<WebDefect> webErrors = StreamEx.of(vr.flattenErrors(pathNodeConverterProvider))
                .map(di -> di.convertPath(PathConverter.prepend(prefixPath)))
                .map(defectPresentationRegistry::getPresentation)
                .map(this::convert)
                .toList();

        List<WebDefect> webWarnings = StreamEx.of(vr.flattenWarnings(pathNodeConverterProvider))
                .map(di -> di.convertPath(PathConverter.prepend(prefixPath)))
                .map(defectPresentationRegistry::getPresentation)
                .map(this::convert)
                .toList();

        result.addErrors(webErrors);
        result.addWarnings(webWarnings);
        return result;
    }

    private WebDefect convert(TranslatableWebDefect translatableWebDefect) {
        Translatable translatable = translatableWebDefect.getTranslatable();
        DefectInfo<? extends Defect> defectInfo = translatableWebDefect.getDefectInfo();
        String translatedText = translationService.translate(translatable);
        return new WebDefect()
                .withCode(defectInfo.getDefect().defectId().getCode())
                .withPath(defectInfo.getPath().toString())
                .withValue(Optional.ofNullable(defectInfo.getValue()).map(Object::toString).orElse(null))
                .withParams(defectInfo.getDefect().params())
                .withText(translatedText)
                .withDescription(translatedText);
    }
}
