/*globals Susanin*/
(function () {
    'use strict';

    var iRoute = null,
        hist = history,
        // параметры, которые нужно отправлять
        LOAD_PARAMS = ['page_type', 'words', 'regions', 'page', 'db'],
        // параметры, которые встречаются на всех страницах
        PUBLIC_PARAMS = ['page_type', 'words', 'regions', 'db'],
        DEFAULTS = {
            db: '',
            page_type: 'words',
            regions: '',
            words: '',
            filter: 'all',
            sort: 'cnt',
            type: 'list',
            period: 'monthly',
            map: 'world',
            page: '1'
        };

    /**
     * Deal with history
     **/
    function HistoryManager() {
        // {Susanin}
        this._route = Susanin.Route({
            pattern: '(#!)(/)(<page_type>)',
            defaults: DEFAULTS
        });
        // {object}
        this._params = {};

        this.refresh();
    }

    HistoryManager.prototype.isDefault = function () {
        var isParamsDefault = true;

        for (var name in this._params) {
            if (DEFAULTS[name] !== undefined && this._params[name] !== DEFAULTS[name]) {
                isParamsDefault = false;
            }
        }

        return isParamsDefault;
    };

    /**
     * @param {object|string} name - pageType if string and there is no value, params if object
     * @param {string|number} [value]
     * @return {object|null} - if hash was changed returns this._params
     **/
    HistoryManager.prototype.set = function (name, value) {
        var params = name;

        if (typeof name === 'string') {
            params = {};
            params[name] = value;
        }

        if (params.words !== undefined && params.words !== this._params.words) {
            this._params.page = DEFAULTS.page;
        }

        this._params = this._sort($.extend(this._params, params));

        this._refreshHash();
        return this._params;
    };

    /**
     * @param {string} [name]
     * @return {object|string}
     **/
    HistoryManager.prototype.get = function (name) {
        if (typeof name === 'string') {
            return this._params[name];
        }
        return $.extend({}, this._params);
    };

    /**
     * @param {string|array:string} nameList
     **/
    HistoryManager.prototype.remove = function (nameList) {
        if (typeof nameList === 'string') {
            nameList = [nameList];
        }
        if (nameList === undefined || nameList.forEach === undefined) {
            return;
        }

        var _this = this;
        function remove(name) {
            delete _this._params[name];
        }
        nameList.forEach(remove);
    };

    HistoryManager.prototype.removePrivat = function (params) {
        var names = PUBLIC_PARAMS;
        if (params === undefined) {
            params = this._params;
        }
        params.page = 1;
        for (var name in params) {
            if (names.indexOf(name) === -1) {
                params[name] = DEFAULTS[name];
            }
        }
        return params;
    };

    /**
     * Добывает параметры из location.hash,
     * возвращает список названий изменившихся полей
     *
     * @retrurn {array:string}
     **/
    HistoryManager.prototype.refresh = function () {
        var href = document.location.href;
        // в библиотеке susanin при парсе + заменяется на пробел
        // для корректной обработки, а нам он нужен
        var hash = document.location.hash.replace(/\+/g, '%2B');

        if (href.indexOf('ncrnd') !== -1) {
            href = href.replace(/\??ncrnd=\d+/, '');
            this._replaceState(href);
        }

        if (href.indexOf('/#%21/') !== -1) {
            href = href.replace(/\/#%21\//, '/#!/');
            hash = hash.replace(/#%21\//, '#!/');
            this._replaceState(href);
        }

        var changedParamsNames = [];
        var hashParams = this._route.match(hash);
        var params = $.extend({}, this._params);

        for (var name in hashParams) {
            if (this._params[name] !== hashParams[name]) {
                changedParamsNames.push(name);
                params[name] = hashParams[name];
            }
            if (typeof params[name] === 'string') {
                params[name] = decodeURIComponent(params[name]);
            }
        }

        this._params = this._sort(params);
        return changedParamsNames;
    };

    /**
     * @return {string|null} - returns hash if it's not like current one
     **/
    HistoryManager.prototype.getRoute = function () {
        var route = this._route.build(this._params);

        if (route.length > 0) {
            route = '#!/' + route;
        } else {
            route = '';
        }

        if (document.location.hash !== route) {
            return route;
        }
        return null;
    };

    HistoryManager.prototype._refreshHash = function () {
        var route = this.getRoute();
        if (route !== null) {
            this._pushState('/' + route);
        }
    };

    /**
     * @param {string} location
     **/
    HistoryManager.prototype._pushState = function (location) {
        if (hist !== undefined && (typeof hist.pushState === 'function')) {
            hist.pushState(null, null, location);
        } else {
            document.location.hash = location.substr(2);
        }
    };

    /**
     * @param {string} location
     **/
    HistoryManager.prototype._replaceState = function (location) {
        if (hist !== undefined && (typeof hist.replaceState === 'function')) {
            hist.replaceState(null, null, location);
        }
    };

    /**
     * Modifies params so that hash always could be the same with same params
     *
     * @param {object} params
     * @return {object}
     **/
    HistoryManager.prototype._sort = function (params) {
        var keys = [];

        for (var key in params) {
            keys.push(key);
        }

        keys.sort();

        function addParam(newParams, key) {
            newParams[key] = params[key];
            return newParams;
        }

        return keys.reduce(addParam, {});
    };

    // block
    BEM.decl('i-route', {
        onSetMod: {
            js: function () {
                if (iRoute === null) {
                    iRoute = this;
                }
                this._initBlock();
                this._addListeners();
                this._refreshRetpath();
            }
        },

        /**
         * @public
         * @param {boolean} hasError
         */
        triggerError: function (hasError) {
            this.trigger('error', {hasError: hasError});
        },

        set: function () {
            var result = this._history.set.apply(this._history, arguments);
            this._refreshRetpath();
            this.trigger('set');
            return result;
        },

        get: function () {
            return this._history.get.apply(this._history, arguments);
        },

        /**
         * Были ли установлены значения отличные от дефолтных
         *
         * @return {boolean}
         **/
        hasParams: function () {
            return !this._history.isDefault();
        },

        remove: function () {
            return this._history.remove.apply(this._history, arguments);
        },

        /**
         * Remove params started from _,
         * this should be used when page_type is changed
         **/
        removePrivat: function () {
            return this._history.removePrivat.apply(this._history, arguments);
        },

        /**
         * @param {object|undefined} params
         * @param {jQuery} renderParams.$container
         * @param {string} renderParams.selector
         **/
        load: function (params, renderParams) {
            if (params) {
                params = this.set(params);
            } else {
                params = this.get();
            }

            this.triggerError(false);
            this._loader.load(params.page_type, params, renderParams);
        },

        _initBlock: function () {
            // {HistoryManager}
            this._history = new HistoryManager();
            // {i-content-loader}
            this._loader = BEM.blocks['i-content-loader'].getInstance();
            // {jQuery}
            this._$win = $(window);
            // {boolean}
            this._firstLoad = true;
        },

        _addListeners: function () {
            if (hist !== undefined && typeof hist.pushState === 'function') {
                this._$win.on('popstate', this._onPopState.bind(this));
            } else {
                //ie 8
                this._$win.on('hashchange', this._onPopState.bind(this));
            }
            this._loader.on('render', this._onRender, this);
        },

        _onRender: function () {
            this._$win.scrollTop(0);
            this.trigger('render');
        },

        _hasLoadParamsChanges: function (paramList) {
            function hasChanges (changes, paramName) {
                if (LOAD_PARAMS.indexOf(paramName) !== -1) {
                    changes = true;
                }
                return changes;
            }

            return paramList.reduce(hasChanges, false);
        },

        _onPopState: function () {
            var changedParamNameList = this._history.refresh();
            this.trigger('refresh');
            // загружаем только если изменились параметры для загрузки
            if (changedParamNameList.length > 0 && this._hasLoadParamsChanges(changedParamNameList)) {
                this.load();
            }
            this._refreshRetpath();
        },

        _refreshRetpath: function () {
            BEM.blocks['i-global'].refreshRetpath();
        }

    }, {
        getInstance: function () {
            if (iRoute === null) {
                iRoute = BEM.create('i-route');
            }
            return iRoute;
        }
    });
})();
