package Exception::API::MediaStorage;
use base qw(Exception);

package QBit::Application::Model::API::Yandex::MediaStorage;

=encoding UTF-8

=cut

use qbit;

use base qw(QBit::Application::Model::API::HTTP);

use MIME::Base64 qw(decode_base64);
use XML::Simple;

sub call {
    my ($self, $url, %params) = @_;

    throw Exception::API::MediaStorage gettext('Expected "url"') unless $url;

    my $headers = HTTP::Headers->new();
    $headers->header('Authorization' => $self->get_option('authorization'),);

    $self->{'__LWP__'}->default_headers($headers) if $headers;

    return $self->SUPER::call($url, %params);
}

=head1 upload

    my $upload_result = $app->api_media_storage->upload('file_name.txt', "file content");

После этого в $upload_result будет что-то вроде:

    {
        complete => [
            {
                addr   => "95.108.229.57:1025",
                group  => 1184,
                path   => "/srv/storage/16/2/data-0.0",
                status => 0,
            },
            {
                addr   => "93.158.130.171:1025",
                group  => 1224,
                path   => "/srv/storage/7/2/data-0.0",
                status => 0,
            },
        ],
        groups   => 2,
        id       => "0:040d67d9b891...7101de95dbd7",
        key      => "1184/file_name.txt",
        obj      => "partner.file_name.txt",
        size     => 12,
        written  => 2,
    }

=cut

sub upload {
    my ($self, $filename, $data, %opts) = @_;

    throw Exception::API::MediaStorage gettext('Expected "filename"') unless $filename;
    throw Exception::API::MediaStorage gettext('Expected "data"')     unless length($data);

    if ($opts{'need_decode'}) {
        $data = decode_base64($data);
        throw Exception::API::MediaStorage gettext("Can't decode base64 string") unless length($data);
    }

    my $namespace = $self->get_option('namespace');
    my $post_port = $self->get_option('post_port');

    my $url = ":$post_port/upload-$namespace/$filename";

    my $xml = $self->call($url, ':post' => TRUE, '' => $data);

    return XMLin($xml);
}

=head1 delete

    $app->api_media_storage->delete("1184/file_name.txt");

Удаляет объект с указнным ключем. В случае ошибки умирает.

=cut

sub delete {
    my ($self, $key) = @_;

    throw Exception::API::MediaStorage gettext('Expected "key"') unless $key;

    my $namespace = $self->get_option('namespace');
    my $post_port  = $self->get_option('post_port');

    my $url = ":$post_port/delete-$namespace/$key";

    return $self->call($url);
}

=head1 statistics

    my $s = $app->api_media_storage->statistics();

И в $s будет нечто вроде:

    {
        bad_couples          => 0,
        broken_couples       => 0,
        closed_couples       => 0,
        effective_free_space => 3708908142901,
        effective_space      => 3708908156028,
        free_space           => 3934190030009,
        frozen_couples       => 0,
        open_couples         => 3,
        removed_keys         => 14,
        total_couples        => 3,
        total_keys           => 40,
        total_space          => 3934190043136,
        uncoupled_groups     => 0,
        uncoupled_space      => 0,
    }

=cut

sub statistics {
    my ($self) = @_;

    my $namespace = $self->get_option('namespace');
    my $get_port  = $self->get_option('get_port');

    my $url = ":$get_port/statistics-$namespace";

    my $json = $self->call($url);

    return from_json($json);
}

sub get_public_link {
    my ($self, $key) = @_;

    throw Exception::API::MediaStorage gettext('Expected "key"') unless $key;

    return $self->get_option('public_url') . '/get-' . $self->get_option('namespace') . "/$key";
}

=head1 get_link_with_upload

    my $link = $app->api_media_storage->get_link_with_upload('the_file.txt', "123");

И в $link будет записан публичный урл загруженного файла:

    "http://storage.mdst.yandex.net/get-partner/1184/the_file.txt"

=cut

sub get_link_with_upload {
    my ($self, $filename, $data, %opts) = @_;

    my $result = $self->upload($filename, $data, %opts);

    return $self->get_public_link($result->{'key'});
}

TRUE;
