package QBit::Git::Branch;

# ABSTRACT: Get info about git branch in QBit project

=encoding UTF-8
=cut

=head1 DESCRIPTION

Модуль для определения git ветки на которую смотрит работчая копия проекта,
сделанного по методологии QBit.

    my $branch = QBit::Git::Branch->new();

    # Получение имени ветки.
    # Может, например, вернуть скаляр с тектом "PI-604_tanker_branches" или,
    # например "v2.5"
    say $branch->get_name();

    # Возвращает true значение если текущая ветка - это ветка для решения
    # задачи, иначе возвращает false.
    # В терминологии git flow ветка для решения задача - это "feature branch"
    # Например, ветка "PI-604_tanker_branches" - это фича ветка
    say $branch->is_feature_branch();

    # Возвращает true значение если текущая ветка - это релизная ветка,
    # иначе возвращает false.
    # Например, ветка "v2.5" - это релизная ветка
    say $branch->is_release_branch();

=cut

use strict;
use warnings FATAL => 'all';
use Carp;

sub new {
    my ($class, @opts) = @_;

    croak "new() should be run without parameters. Stopped" if @opts;

    my $self = {};

    my $output = `/usr/bin/git branch --no-color`;
    my @lines = split /\n/, $output;

    foreach (@lines) {
        if ($_ =~ /^\* (.+)$/) {
            $self->{_name} = $1;
            last;
        }
    }

    croak "Can't find out git branch. Stopped" if not defined $self->{_name};

    bless $self, $class;
    return $self;
}

sub print {
    my ($self) = @_;

    print "Current git branch is " . $self->get_name() . ".\n";
}

sub get_name {
    my ($self) = @_;

    return $self->{_name};
}

sub is_feature_branch {
    my ($self) = @_;

    my $result = not $self->is_release_branch();
    return $result;
}

sub is_release_branch {
    my ($self) = @_;

    my $result = $self->{_name} =~ /^v2\.\d+$/;
    return $result;
}

1;
