package QBit::Tanker;

use strict;
use warnings FATAL => 'all';
use Carp;
use Term::ANSIColor qw(colored);
use utf8;

use JSON;
use Locale::PO;
use Yandex::Tanker;
use QBit::Git::Branch;

# ABSTRACT: QBit Framework + Yandex.Tanker = ♥

=begin comment

This class should be used only in bin/qbit_tanker_* scripts.

All the methods start with "__" to show that they are private and should not
be used elewhere.

=end comment

=cut

my $true  = 1;
my $false = '';

sub new {
    my ($class, %params) = @_;

    if (%params) {
        croak "Error. You don't need to pass any parameters to the constructor.";
    }

    my $self = {};
    bless $self, $class;

    $self->{__config} = $self->__get_config_data();

    return $self;
}

=begin comment __get_config_data

This method returns hashref with the data from config file `qbit_tanker.conf`.
The config file should be stored in the directory from where the qbit_tanker_*
scripts are run.

In case of any error the script will die.

Here is the example data that this method returns:

    {
        lang    => [
            "en",
            "ru",
        ],
        project => "tanker_project_id",
        server  => "test",
        token   => "secret",
    }

TODO bes

 * croak in case some obligatory are not specified

=end comment

=cut

sub __get_config_data {
    () = @_;

    my $config_data;

    my $filename = 'qbit_tanker.conf';

    if (not -e $filename) {
        croak "File '$filename' not found. You should run script $0 in the directory with this file. Stopped";
    }

    my @known_parametes = qw(
      project
      token
      server
      lang
      mo_filename
      msgid_dir
      );

    my $config_file_content = __read_file(undef, $filename);

    foreach my $config_file_line (split /\n/, $config_file_content) {

        next if $config_file_line =~ /^\s*$/;
        next if $config_file_line =~ /^\s*#/;

        my $line_is_correct = $config_file_line =~ /
            ^
            \s*
            (\S+)   # parameter name
            \s*
            =
            \s*
            (\S+)   # parameter value
            \s*
            $
        /x;

        if ($line_is_correct) {
            my $key   = $1;
            my $value = $2;

            my $parameter_is_known = grep {$_ eq $key} @known_parametes;

            if ($parameter_is_known) {

                if ($key ne 'lang' and $key ne 'msgid_dir' and not defined $config_data->{$key}) {
                    $config_data->{$key} = $value;
                } elsif ($key eq 'lang' or $key eq 'msgid_dir') {
                    push @{$config_data->{$key}}, $value;
                } else {
                    croak "Error in config file '$filename'. Paramter '$key' is used too many times";
                }

            } else {
                croak "Unknown paramter '$key' in config file '$filename'. Stopped";
            }

        } else {
            croak "Incorrect line in config file '$filename':\n" . $config_file_line . "\nStopped";
        }
    }

    for my $p (qw/project token server/) {
        my $env_key = "QBIT_TANKER_".uc($p);
        if ($ENV{$env_key}) {
            $config_data->{$p} = $ENV{$env_key};
        } elsif ($ENV{$env_key."_PATH"}) {
            ($config_data->{$p} = __read_file(undef, $ENV{$env_key."_PATH"})) =~ s/^\s+|\s+$//g;
        }
    }

    return $config_data;
}

sub __get_branches {
    my ($self) = @_;

    return $self->__get_tanker()->get_branches();
}

sub __create_branch_if_needed {
    my ($self, $branch_name, $ref_branch_name) = @_;

    my @tanker_branches = $self->__get_branches();

    if ($ref_branch_name && not(grep {$_ eq $ref_branch_name} @tanker_branches)) {
        croak "Error. There is no reference branch '$ref_branch_name' in tanker. Stopped";
    }

    if (not(grep {$_ eq $branch_name} @tanker_branches)) {
        print "There is no branch " . $branch_name . " in tanker. Need to create.\n\n";

        unless ($ref_branch_name) {
            $self->__print_branches();

            $ref_branch_name =
              $self->__get_tanker_branch_from_user("Enter the branch that should be base for the new branch");
        }

        print "Creating tanker branch " . colored($branch_name, "blue") . " based on $ref_branch_name\n";

        my $script_config = $self->__get_script_config_with_token();

        system("tanker_branch $script_config --create=$branch_name --ref=$ref_branch_name");
    } else {
        # There is already branch in tanker. Doing noting.
    }

}

=begin comment __download_all_po_from_tanker

It will create such a structure with po files:

    locale/
    └── tmp
        ├── en.po
        ├── ru.po
        └── tr.po

=end comment

=cut

sub __download_all_po_from_tanker {
    my ($self, $branch_name) = @_;

    my $langs = join ",", $self->__get_langs();
    my $script_config = $self->__get_script_config_without_token();

    `mkdir -p locale/`;

    system(
            "tanker_download" . " "
          . $script_config
          . " --status=unapproved"
          . " --lang=$langs"

          # Не удаляем файлы locale/tmp/*.po, чтобы их можно было смерждить
          # с po файлами из фреймворка
          . " --dont_delete_tmp_files"

          # mo файл будет скомпилирован после объединиения po файлов, так
          # что на этом этапе компилировать их не нужно
          . " --dont_compile_mo"

          # В танкере из-за merge может случится ситуация что один и тот же
          # msgid будет продублирован в разных кейсетах. В этом случае po
          # файл, выгружаемый из танкера будет битым (по стандарту в po файле
          # msgid не может дублироваться). Поэтому правим po файл, который мы
          # получаем из танкера.
          . " --fix_po_from_tanker"

          . " --branch=$branch_name"
    );

    return $false;
}

sub __get_tanker {
    my ($self) = @_;

    if (!defined($self->{'__tanker'})) {
        $self->{'__tanker'} = Yandex::Tanker->new(
            {
                url     => $self->__get_tanker_url(),
                project => $self->{__config}->{project},
                token   => $self->{__config}->{token},
            }
        );
    }

    return $self->{'__tanker'};
}

sub __get_tanker_url {
    my ($self) = @_;

    my $url;
    my $server = $self->{__config}->{server};

    if ($server eq 'prod') {
        $url = $Yandex::Tanker::PROD_URL;
    } elsif ($server eq 'test') {
        $url = $Yandex::Tanker::TEST_URL;
    } else {
        croak "Error. Can't find out tanker server url. Stopped";
    }

    return $url;
}

sub __read_file {
    my (undef, $filename) = @_;

    open FILE, "< :encoding(UTF-8)", $filename or croak "Can't open file '$filename' to read: $!. Stopped";
    my @lines = <FILE>;
    close FILE;

    my $content = join '', @lines;

    return $content;
}

sub __write_file {
    my (undef, $filename, $content) = @_;

    open FILE, "> :encoding(UTF-8)", $filename or croak "Can't open file '$filename' to write: $!. Stopped";
    print FILE $content;
    close FILE;

    return $false;
}

=head2 __get_script_config_with_token # TODO bes - translate pod to english

B<Get:> 1) $self

B<Return:> 1) $config - скаляр с параметрами, которые нужно передавать
скриптам из пакета libyandex-tanker-perl.

Этот метод берет данные из конфиг файла lib/Application.cfg и из них формирует
строку, которую нужно передавать в скрипты.

Например, этот метод может вернуть:

    "--project=pi --token=123 --use_test_tanker"

=cut

sub __get_script_config_with_token {
    my ($self) = @_;

    my $config_string = "--project=" . $self->{__config}->{project} . " --token=" . $self->{__config}->{token};

    if ($self->{__config}->{server} eq "test") {
        $config_string .= " --use_test_tanker";
    }

    return $config_string;
}

=head2 __get_script_config_without_token # TODO bes - translate pod to english

То же самое что и get_script_config_with_token, но в скалере который
возвращается нет токена.

Например, этот метод может вернуть:

    "--project=pi --use_test_tanker"

=cut

sub __get_script_config_without_token {
    my ($self) = @_;

    my $config_string = "--project=" . $self->{__config}->{project};

    if ($self->{__config}->{server} eq "test") {
        $config_string .= " --use_test_tanker";
    }

    return $config_string;
}

sub __get_langs {
    my ($self) = @_;

    my @langs = @{$self->{__config}->{lang}};

    return @langs;
}

sub __merge_po {
    my ($self) = @_;

    my @langs = $self->__get_langs;

    `mkdir -p locale/merged`;

    foreach (@langs) {
        my $cmd       = "msgmerge ./locale/tmp/$_.po ./locale/tmp/strings.po -N";
        my $po_merged = `$cmd`;

        if (length($po_merged) > 0) {
            utf8::decode($po_merged);
            $self->__write_file("./locale/merged/$_.po", $po_merged);
        } else {
            croak "Error. `msgmerge` does not returned po. Stopped";
        }
    }

    return $false;
}

sub __compile_mo {
    my ($self, $lang) = @_;

    my $locale      = $self->__get_locale($lang);
    my $mo_filename = $self->{__config}->{mo_filename};

    `mkdir -p locale/$locale/LC_MESSAGES/`;
    `msgfmt --use-fuzzy locale/merged/$lang.po -o locale/$locale/LC_MESSAGES/$mo_filename.mo`;

    return $false;
}

sub __rm_tmp_files {
    my ($self) = @_;

    `rm -rf ./locale/merged ./locale/tmp ./locale/keysets`;

    return $false;
}

sub __get_locale {
    my ($self, $lang) = @_;

    my %known_locales = (
        be => 'be_BY',
        en => 'en_GB',
        kk => 'kk_KZ',
        ru => 'ru_RU',
        tr => 'tr_TR',
        uk => 'uk_UA',
    );

    my $locale;

    if (exists $known_locales{$lang}) {
        $locale = $known_locales{$lang};
    } else {
        croak "Error. QBit::Tanker does not know locale for lang '$lang'. Stopped";
    }

    return $locale;
}

sub __print_branches {
    my ($self) = @_;

    $self->__print_some_branches($self->__get_branches());

    return $false;
}

sub __print_some_branches {
    my ($self, @branches) = @_;

    print "Here is the list of branches in tanker:\n\n";

    foreach (@branches) {
        print " * $_\n";
    }

    return $false;
}

sub __get_tanker_branch_from_user {
    my ($self, $message) = @_;

    my @tanker_branches = $self->__get_branches();

    my $branch_name = '';

    while (not(grep {$_ eq $branch_name} @tanker_branches)) {
        print "\n$message? ";
        $branch_name = <>;
        chomp $branch_name;
    }

    return $branch_name;
}

sub __extract_strings_to_po {
    my ($self) = @_;

    `mkdir -p ./locale/tmp`;

    my $po_file = "./locale/tmp/strings.po";

    my $framework_dir = "framework_yndx/lib/";

    my $cmd =
        "qbit_xgettext"
      . join('', map {" --dir $_/"} @{$self->{__config}->{msgid_dir}})
      . " --extractor 'QBit::Gettext::Extract::Lang::JS=.i18n_list'"
      . " --ofile $po_file";

    system $cmd and die "Couldn't run: $cmd ($!)";

    return $false;
}

sub __split_po_to_keysets {
    my ($self) = @_;

    my $keysets_dir = "./locale/keysets";

    `mkdir -p $keysets_dir`;

    my @langs = $self->__get_langs;

    foreach my $lang (@langs) {
        my $po_file = "./locale/merged/$lang.po";

        my $header;
        my %keysets;

        foreach my $element (@{Locale::PO->load_file_asarray($po_file)}) {
            if ($element->msgid() eq '""') {
                $header = $element;
            } elsif (!$element->obsolete) {
                $element->reference() =~ /^(.*?):\d+/;
                my $keyset = $1;
                $keyset =~ s{/}{__}g;
                push @{$keysets{$keyset}}, $element;
            }
        }

        my $keysets_dir_lang = $keysets_dir . "/$lang";

        `mkdir -p $keysets_dir_lang`;
        foreach my $keyset (sort keys %keysets) {
            Locale::PO->save_file_fromarray($keysets_dir_lang . "/" . $keyset . ".po", [$header, @{$keysets{$keyset}}],
            );
        }
    }

    return $false;
}

sub __upload_keysets {
    my ($self, $branch_name) = @_;

    croak "Expected to recive 'branch'. Stopped" if not defined $branch_name;

    my $langs = join ",", $self->__get_langs();
    my $script_config = $self->__get_script_config_with_token();

    system("tanker_upload $script_config --lang=$langs --dont_write_original --branch=$branch_name");

    return $false;
}

sub __merge_in_tanker {
    my ($self, %params) = @_;

    croak "Expected 'from'. Stopped" if not defined $params{from};
    croak "Expected 'to'. Stopped"   if not defined $params{to};

    my $script_config = $self->__get_script_config_with_token();

    my $exit_status = system(
"tanker_merge $script_config --source=$params{from} --dest=$params{to} --use_references_from_dest --auto_resolve_simple_case"
    );
    croak "tanker_merge failed. Stopped" if $exit_status != 0;

    return $false;
}

sub __delete_tanker_branch_if_needed {
    my ($self, $branch_name) = @_;

    print "\nDo you want to delete tanker branch " . colored($branch_name, "blue") . "? [y/N] ";
    my $answer = <>;
    chomp $answer;

    if (lc $answer eq 'y') {
        my $script_config = $self->__get_script_config_with_token();
        system("tanker_branch $script_config --delete=$branch_name");
    }

    return $false;
}

sub __get_tanker_branch_to_merge_to {
    my ($self, $source_branch_name, $dest_branch_name) = @_;

    if ($dest_branch_name) {
        if (not(grep {$_ eq $dest_branch_name} $self->__get_branches())) {
            croak "Error. There is no destination branch '$dest_branch_name' in tanker. Stopped";
        }
    } else {
        $self->__print_branches();

        $dest_branch_name =
          $self->__get_tanker_branch_from_user("Enter the branch where to merge tanker branch '$source_branch_name'");
    }

    return $dest_branch_name;
}

sub __create_js {
    my ($self, $lang) = @_;

    my $po_file = "locale/merged/$lang.po";

    # $block_strings will containg something like:
    #
    #    {
    #        blocks/b-statistic-chart   => {
    #            "no-data-to-display"       => "Нет данных для отображения.",
    #            "too-much-data-to-display" => "Слишком много данных для отображения.",
    #        },
    #        blocks/b-wizard-form       => {
    #            "loading-please-wait" => "Идёт загрузка, пожалуйста, подождите",
    #        },
    #    }
    my $block_strings;

    foreach my $element (@{Locale::PO->load_file_asarray($po_file)}) {

        my $reference = $element->reference();

        if (defined($reference)) {

            my ($msgid) = $element->msgid() =~ /"(.*)"/;

            my $msgstr = $self->__get_msgstr($element->msgstr());
            $msgstr = $msgid if $msgstr eq '';

            my @lego_blocks = $self->__get_lego_blocks($reference);
            $block_strings->{$_}->{$msgid} = $msgstr foreach @lego_blocks;

        }
    }

    $self->__write_js_files(
        lang           => $lang,
        tanker_strings => $block_strings,
    );

    return $false;
}

=begin comment __write_js_files

Input: ($self, %params)

Output: -

Example of %params :

    (
        lang    => "ru",
        tanker_strings => {
            blocks/b-chart                     => {
                loading                          => "Загрузка...",
                month-apr                        => "Апрель",
                month-apr-short                  => "Апр",
                month-aug                        => "Август",
                month-aug-short                  => "Авг",
                month-dec                        => "Декабрь",
                month-dec-short                  => "Дек",
                month-feb                        => "Февраль",
            },
            blocks/b-clipboard                 => {
                'Copy to clipboard' => "Скопировать в буфер",
            },
    )

=end comment

=cut

sub __write_js_files {
    my ($self, %params) = @_;

    foreach my $block (keys %{$params{tanker_strings}}) {

        $block =~ m{.*/(.*?)$};
        my $block_name = $1;

        my $dir           = $block . "/" . $block_name . ".i18n/";
        my $list_filename = $block . "/" . $block_name . ".i18n_list";
        my $filename      = $dir . $params{lang} . ".js";

        next if !-f $list_filename;

        my $all_strings_file_name = $block . "/" . $block_name . ".i18n_list";
        my @all_block_strings     = @{from_json $self->__read_file($all_strings_file_name)};

        my $po_content = $self->__create_js_content(
            block_name     => $block_name,
            tanker_strings => $params{tanker_strings}->{$block},
            all_strings    => \@all_block_strings,
        );

        `mkdir -p $dir`;

        $self->__write_file($filename, $po_content);

    }

    return $false;
}

=begin comment __create_js_content

Input: ($self, %params)

Output: $string with data in special LEGO format, that should be saved to the
file b-block-name/b-block-name.i18n/lang.js

Example of %params :

    (
        block_name     => "b-media-brands",
        tanker_strings => {
            'Has invalid data' => "Есть некорректно заполненные поля",
            Save               => "Сохранить",
        },
        all_strings    => [
            "Media brands description",
            "Input hint",
            "Has invalid data",
            "Save",
        ],
    )

=end comment

=cut

sub __create_js_content {
    my ($self, %params) = @_;

    my $strings = $params{tanker_strings};
    foreach my $string (@{$params{all_strings}}) {
        if (not exists $strings->{$string}) {
            $strings->{$string} = $string;
        }
    }

    my $data = {$params{block_name} => $strings};
    my $json = to_json($data, {pretty => $true});
    my $content =
        "/* Это автоматически сгенерированный файл. */\n\n"
      . "module.exports = "
      . $json;

    chomp $content;
    $content .= ";\n";

    return $content;
}

sub __get_msgstr {
    my ($self, $msgstr) = @_;

    return '' if !defined($msgstr);

    my ($result) = $msgstr =~ /"(.*)"/;
    $result =~ s{\\"}{"}g;
    utf8::decode($result);
    return $result;
}

sub __get_lego_blocks {
    my ($self, $reference) = @_;

    my @blocks;
    while ($reference =~ m{LEGO/(.*?):}mg) {
        push @blocks, $1;
    }

    return @blocks;
}

1;
