package Wordstat::Application::Model::WsCaptcha;

use base qw(QBit::Application::Model);

use qbit;

use Digest::MD5 qw(md5_hex);

our $LS_COOKIE_NAME = 'wsst'; # Word Stat STatus
our $LS_COOKIE_SALT = 'Oh... So many bugs';
our $LS_COOKIE_INTERVAL_LIMIT = 24 * 60 * 60; # in sec

=head2 last_success_is_get

Получает дату последней успешной проверке капчи.

Для этого вычисляет uid (из request) с желаемым форматом (из текущей кукуи) и валидирует куку

Вход:

  * request - ref, для вычисления uid и получения кукуи

Выход:

  * FALSE в случае ошибки
  * \d, дата события, в сек.

Икслючения: -

=cut
sub last_success_get {
    my ($self, %opts) = @_;
    my ($request) = @opts{qw/request/};

    # get cookie (and simple, fast check)
    my $cookie = $request->cookie($LS_COOKIE_NAME);
    return FALSE unless $cookie;
    my (undef, $c_ver, $c_date) = $cookie =~/^(.{32})(.{1})\|(.{8})$/;
    return FALSE unless defined($c_ver) && defined($c_date);
    # check maximum limit
    return FALSE if $c_date && curdate(oformat => 'sec') - hex($c_date) > $LS_COOKIE_INTERVAL_LIMIT;
    # get uid and simple check
    my $uid = $self->last_success_compute_uid(request => $request, ver => $c_ver);
    return FALSE unless $uid;
    # validate cookie
    return hex($c_date) if $self->_last_success_data_sign(key => $uid, data => [$c_ver, $c_date]) eq $cookie;
    return FALSE;
}

=head2 last_success_set

Сохраняем информацию (дату) об успешной капче (в качестве хранилища - подписанная кука)

Вход:

  * request - ref, для вычисления uid
  * response - ref, для сохранения куки и для формирования uid пользователя
  * [date] - \d, дата события, в сек., по умолчанию - сейчас

Выход: -

  * создает куку $LS_COOKIE_NAME ('wsst') = <sign_key:32><ver:1><create_date_hex:8>

Икслючения: -

=cut
sub last_success_set {
    my ($self, %opts) = @_;
    my ($request, $response, $date) = @opts{qw/request response date/};
    my $uid = $self->last_success_compute_uid(request => $request);
    return unless $uid;
    my $ver = substr($uid, 0, 1); # 1
    my $date_hex = unpack("H*", pack("N", $date || curdate(oformat => 'sec'))); # 8
    $response->add_cookie(
        $LS_COOKIE_NAME => $self->_last_success_data_sign(key => $uid, data => [$ver, $date_hex]),
        # domain => '.yandex.ru',
        expires => '+1h',
    ) if $uid;
}

=head2 _last_success_data_sign

Подписываем данные ключем. На выходе строка из ключа и данных через "|"

Вход:

  * key - $, строка, используемая для подписи
  * data - [], подписываемые данные,

Выход:

  $, <sign - md5_hex:32b><data1:?>|<data2:?>|<data3:?>

Исключания:

  'Wrong Key or Data' - неверные параметры

# <md5_hex:32><data1:?><data2:?>...
=cut
sub _last_success_data_sign {
    my ($self, %opts) = @_;
    my ($key, $data) = @opts{qw/key data/};
    throw 'Wrong Key or Data' unless $key && $data && ref($data) eq 'ARRAY';

    return
        md5_hex(
            join('', @$data)
            . $LS_COOKIE_SALT
            . $key
        )
        . join('|', @$data);
}


=head2 last_success_compute_uid

Вычисляем уникальный ID пользователя. В принципе - несколько алгоритмов.

На выходе строка с информацией об алгоритме выборки.

Вход:

  * request - ref, объект из которого выуживаем информация
  * [ver] - \d, номер формата. Опционально желаемый

Выход:

  * undef - никак нельзя узнать кто это
  * <ver:1><rest_of_uid:?>

Икслючения: -

Форматы:

  1 - по "remote_addr" и куке "fuid01"
  2 - по "remote_addr" и логину


=cut
sub last_success_compute_uid {
    my ($self, %opts) = @_;
    my ($ver, $request) = @opts{qw/ver request/};
    my $cur_user = $self->get_option('cur_user');

    # choose best way
    unless (defined $ver) {
        if ($cur_user->{login}) {
            $ver = 2;
        } elsif ($request->cookie('fuid01')) {
            $ver = 1;
        } else {
            $ver = 0;
        }
    };
    # get uid
    if ($ver == 1 && $request->cookie('fuid01')) {
        return join('|', $ver, $request->remote_addr, $request->cookie('fuid01') // '');
    } elsif ($ver == 2 && $cur_user->{login}) {
        return join('|', $ver, $request->remote_addr, $cur_user->{login} // '');
    } else {
        return;
    }
}


TRUE;
