package Yandex::Shell;

# $Id: Shell.pm 3007 2013-11-27 12:50:35Z andy-ilyin $

=head1 NAME

    Yandex::Shell

=head1 DESCRIPTION

    "Безопасный" вызов внешних программ:
        - умеет правильно квотить параметры
        - умирает в случае неудач
        - прозрачно работает с временными файлами

=head1 SYNOPSIS

    yash_system("perl", "unit_tests/runtests.pl", "--all");

    my $ret = yash_qx("echo", "test");
    
    my $quoted = yash_quote("<asdfsdaf; asdfasdf>");

    my $prog = "print 1";
    yash_system("perl", \$prog);

    yash_system("perl", {ref => \$prog});

    yash_system("wget", "-O", {unlink => 1, ref => \$cont}, "http://ya.ru/");

=head1 PARAMETERS PROCESSING

    В качестве параметров можно передать:
        скаляр - он заквотится и передастся вызываемой программе
        ссылка на скалар - текст запишется во временный файл и программе передастся имя файла.
                    после того как программа отработает, данные из файла прочитаются обратно в скаляр
        ссылка на хэш, возможные ключи:
            ref - ссылка на скаляр, поведение описано в предыдущем пункте
            unlink - булевское значение, нужно ли созданный временный файл удалить перед выполнением программы

=cut

use strict;
use warnings;

use File::Temp qw/tempfile tempdir/;
use File::Path qw/rmtree/;
use File::Slurp;
use Cwd qw/cwd/;

use base qw/Exporter/;
our @EXPORT = qw/
    yash_quote
    yash_system
    yash_qx
    /;

=head1 PACKAGE VARIABLES

=head2 $Yandex::Shell::CHDIR

    Булевская переменная, определяющая, нужно ли для выполнения программы 
    создавать временную директорию и переходить в неё.
    По-умолчанию - 0

=cut

our $CHDIR = 0;

=head2 $Yandex::Shell::PRINT_COMMANDS

    Булевская переменная, определяющая, нужно ли выводить на экран (в STDERR)
    команды перед их выполнением.

=cut

our $PRINT_COMMANDS = 0;

=head1 PACKAGE FUNCTIONS

=head2 yash_quote('text param')

    Заквотить строку для передачи в шелл.

=cut
sub yash_quote {
    my $param = shift;
    if ($param =~ /^[\w:\/\-=\.]+$/) {
        return $param;
    }
    $param =~ s/(["\$\\!])/\\$1/g;
    return "\"$param\"";;
}

=head2 yash_system($command, @params)

    Выполнить программу с указанными параметрами.
    В случае ошибки - die.
    Ничего не возвращает.

=cut
sub yash_system {
    my ($fhs, $params) = _process_data_params(@_);
    my $cmd = join(' ', map {yash_quote($_)} @$params);

    _before_exec();
    print STDERR "running: $cmd\n" if $PRINT_COMMANDS;
    my $res = system(@$params);
    _after_exec();
    _clear_tmp_files($fhs);

    if ($res != 0) {
        die "system call '$cmd' failed\n";
    }
}

=head2 yash_qx($command, @params)

    Выполнить программу с указанными параметрами.
    В случае ошибки - die.
    Возвращает STDOUT программы.

=cut
sub yash_qx {
    my ($fhs, $params) = _process_data_params(@_);
    my ($prog, @args) = @$params;
    my $cmd = join(' ', $prog, map {yash_quote($_)} @args);

    _before_exec();
    print STDERR "running: $cmd\n" if $PRINT_COMMANDS;
    my $pid = open(my $fh, "-|", $cmd);
    if (!$pid) {
        _after_exec();
        _clear_tmp_files($fhs);
        die "Can't open '$cmd': $!";
    }
    my $buffer = join '', <$fh>;
    my $ret = waitpid($pid, 0);

    _after_exec();
    _clear_tmp_files($fhs);

    if ($ret != $pid) {
        die "Error: can't wait child '$cmd'";
    }
    if ($?) {
        die "Error: child return error: $?, $!";
    }
    return $buffer;
}

sub _process_data_params {
    my @fhs;
    my @params = map {
        if (ref $_) {
            my ($fh, $file) = tempfile();
            my $dataref = ref $_ eq 'SCALAR' ? $_ : $_->{ref};
            if (ref $_ eq 'HASH' && $_->{unlink}) {
                unlink $file or die "Can't unlink $file: $!";
            }
            if (defined $$dataref) {
                print $fh $$dataref or die "Can't write to $file: $!";
            }
            push @fhs, [$fh, $file, $dataref];
            $file;
        } else {
            $_;
        }
    } @_;
    return \@fhs, \@params;
}

sub _clear_tmp_files {
    my $fhs = shift;
    for my $fh (@$fhs) {
        ${$fh->[2]} = read_file($fh->[1]) if -f $fh->[1];
        unlink $fh->[1];
    }
}

{
    my ($cwd, $tmpdir);
    sub _before_exec {
        if ($CHDIR) {
            $cwd = cwd();
            $tmpdir = tempdir();
            if (!chdir($tmpdir)) {
                rmtree($tmpdir);
                die "Can't chdir to $tmpdir: $!";
            }
        } else {
            $cwd = $tmpdir = undef;
        }
    }

    sub _after_exec {
        if ($CHDIR && $cwd && $tmpdir) {
            chdir($cwd) || die "Cant chdir to $cwd: $!";
            rmtree($tmpdir);
        }
        $cwd = $tmpdir = undef;
    }
}

1;
