package ru.yandex.chemodan.app.idmapi;

import java.util.function.BiFunction;

import lombok.AllArgsConstructor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.boot.admin.idm.UserAccess;
import ru.yandex.chemodan.boot.admin.idm.UsersRegistry;
import ru.yandex.chemodan.util.idm.IdmAllRolesResponse;
import ru.yandex.chemodan.util.idm.IdmInfoResponse;
import ru.yandex.chemodan.util.idm.IdmOkResponse;
import ru.yandex.chemodan.util.idm.IdmResponse;
import ru.yandex.chemodan.util.idm.IdmRoutines;
import ru.yandex.chemodan.util.idm.MultilingualString;
import ru.yandex.chemodan.util.idm.RoleDescription;
import ru.yandex.chemodan.util.idm.RoleNode;
import ru.yandex.chemodan.util.idm.RoleSpecification;
import ru.yandex.chemodan.util.idm.SlugNode;
import ru.yandex.commune.alive2.AliveAppInfo;
import ru.yandex.commune.alive2.AliveAppsHolder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author swined
 */
@AllArgsConstructor
public class IdmRegistriesRoutines implements IdmRoutines {

    private static final String ROOT_KEY = "apps";

    private static final Logger logger = LoggerFactory.getLogger(IdmRegistriesRoutines.class);

    private static final MultilingualString APPS = MultilingualString.bilingual("Applications", "Приложения");
    private static final MultilingualString ROLES = MultilingualString.bilingual("Roles", "Роли");

    private final AliveAppsHolder aliveAppsHolder;
    private final UsersRegistry usersRegistry;

    private static String getAppId(AliveAppInfo info) {
        return String.format("%s-%s", info.getServiceName(), info.getAppName());
    }

    private ListF<String> getApps() {
        return aliveAppsHolder
                .aliveApps()
                .groupBy(IdmRegistriesRoutines::getAppId)
                .keys()
                .plus(
                        "dataapi-dataapi",
                        "dataapi-dataapi-worker",
                        "docviewer-web",
                        "telemost-telemost-backend",
                        "telemost-tcm"
                );
    }

    @Override
    public IdmInfoResponse getInfo() {
        SlugNode perms = SlugNode.definitive(SlugNode.ROLE_KEY, ROLES, Cf.list(Role.values()).map(role -> RoleDescription.bilingual(role.name(), role.en, role.ru)));
        return RoleNode.root(new SlugNode(ROOT_KEY, APPS, getApps().toMap(Function.identityF(), app -> RoleNode.intermediate(MultilingualString.singleValue(app), perms)))).asIdmInfo();
    }

    private IdmResponse update(String login, RoleSpecification spec, BiFunction<SetF<String>, String, SetF<String>> func) {
        ListF<String> values = spec.getValues(Cf.list(IdmRegistriesRoutines.ROOT_KEY, SlugNode.ROLE_KEY));
        usersRegistry.doSynchronized(() -> {
            UserAccess access = usersRegistry.getO(login).orElseGet(() -> UserAccess.empty(login));
            usersRegistry.put(Role.parse(values.last()).map(role -> access.update(values.first(), role.name(), func))
                    .orElse(access));
        });
        return new IdmOkResponse();
    }

    @Override
    public IdmResponse addRole(String login, RoleSpecification role) {
        logger.info("addRole " + login + " " + role);
        return update(login, role, SetF::plus1);
    }

    @Override
    public IdmResponse removeRole(String login, RoleSpecification role, boolean fired) {
        logger.info("removeRole " + login + " " + role);
        return update(login, role, SetF::minus1);
    }

    @Override
    public IdmAllRolesResponse getAllRoles() {
        return new IdmAllRolesResponse(usersRegistry.getAll().map(ua -> ua.getUserRoles(IdmRegistriesRoutines::getRoleSpec)));
    }

    private static RoleSpecification getRoleSpec(String app, String role) {
        return new RoleSpecification(Tuple2List.<String, String>tuple2List()
                .plus1(IdmRegistriesRoutines.ROOT_KEY, app)
                .plus1(SlugNode.ROLE_KEY, role)
        );
    }

}
