package ru.yandex.chemodan.app.idmapi;

import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.boot.admin.idm.AppAccess;
import ru.yandex.chemodan.boot.admin.idm.AppsRegistry;
import ru.yandex.chemodan.boot.admin.idm.UserAccess;
import ru.yandex.chemodan.boot.admin.idm.UsersRegistry;

@SuppressWarnings("unused")
public class IdmTools {

    private final AppsRegistry appsRegistry;
    private final UsersRegistry usersRegistry;

    public IdmTools(AppsRegistry appsRegistry, UsersRegistry usersRegistry) {
        this.appsRegistry = appsRegistry;
        this.usersRegistry = usersRegistry;
        putDefault();
    }

    public void addDefaultAppRole(String app, Role role) {
        appsRegistry.update(app, access -> access.getDefaultRoles().add(role.name()));
    }

    public void removeDefaultAppRole(String app, String role) {
        appsRegistry.update(app, access -> access.getDefaultRoles().remove(role));
    }

    public void addAppRole(String app, String path, Role role) {
        appsRegistry.update(app, access -> access.getRoleMap().computeIfAbsent(path, k -> Cf.hashSet()).add(role.name()));
    }

    public void removeAppRole(String app, String path, String role) {
        appsRegistry.update(app, access -> access.getRoleMap().computeIfAbsent(path, k -> Cf.hashSet()).add(role));
    }

    public void removeAppPath(String app, String path) {
        appsRegistry.update(app, access -> access.getRoleMap().removeO(path));
    }

    private void putDefault() {
        appsRegistry.awaitInitialization();
        appsRegistry.put(defaultAccess());
    }

    private static AppAccess defaultAccess() {
        return new AppAccess(
                "default",
                Cf.set(Role.READ.name()),
                Cf
                        .map("/", Role.READ)
                        .plus1("/script", Role.EXECUTE)
                        .plus1("/scripter", Role.SCRIPTER)
                        .plus1("/repl", Role.EXECUTE)
                        .plus1("/manager", Role.EXECUTE)
                        .plus1("/user-databases", Role.DATASYNC)
                        .plus1("/external-databases", Role.DATASYNC)
                        .plus1("/app-settings", Role.DATASYNC)
                        .plus1("/access-rate-limits-settings", Role.DATASYNC)
                        .plus1("/uaas", Role.UAAS)
                        .plus1("/zk", Role.ZK)
                        .plus1("/zk/update", Role.ZK_WRITE)
                        .plus1("/zk/delete", Role.ZK_WRITE)
                        .plus1("/telemost-configurations", Role.TELEMOST_CLIENT_CONFIGURATION)
                        .plus1("/tcm", Role.EXECUTE)
                        .plus1("/care", Role.CARE)
                        .plus1("/promo", Role.PROMO)
                        .plus1("/partner", Role.PARTNER)
                        .mapValues(role -> Cf.set(role.name()))
        );
    }

    /*
     * To call from console
     *
     * login = "smdenis"
     * app = "disk-notifier"
     * role = "READ"
     *
     * usersRegistry.update(login, function(ua) {return ua.update(app, role, function(a, b) {return a.plus1(b)})});
     *
     */
    public void addUserRole(String login, String app, Role role) {
        usersRegistry.update(login, ua -> ua.update(app, role.name(), SetF::plus1));
    }

    public void removeUserRole(String login, String app, String role) {
        usersRegistry.update(login, ua -> ua.update(app, role, SetF::minus1));
    }

    public Map<String, Collection<String>> inconsistencies() {
        return usersRegistry.getAll().toMap(UserAccess::getLogin, this::getInconsistentRoles).filterValues(i -> !i.isEmpty());
    }

    public Collection<String> getInconsistentRoles(UserAccess access) {
        return access
                .getRoles()
                .values()
                .map(Cf::x)
                .reduceLeftO(SetF::plus)
                .getOrElse(Cf::set)
                .stream()
                .filter(role -> !Role.parse(role).isPresent())
                .collect(Collectors.toSet());
    }

    public void fixRoleCase() {
        usersRegistry.getAll().map(ua -> ua.remapRoles(r -> Cf.list(Role.values()).filter(x -> x.name().equalsIgnoreCase(r)).firstO().map(Role::name).getOrElse(r))).forEach(usersRegistry::put);
    }


}
