import os
import sys

import yaml

from .logger import EnvironmentLogger


class BaseConfig:
    environment_mapping = {}
    _default_config = {}

    def __init__(self, config={}):
        config = self.load_config(config)
        self._config = self.parse_environment(config)
        self.after_init()

    @staticmethod
    def load_config(config={}):
        return config

    def after_init(self):
        pass

    @classmethod
    def parse_environment(cls, config):
        for env, key in cls.environment_mapping.items():
            if env in os.environ:
                config[key] = os.environ[env]
        return config

    def __getattr__(self, key):
        return self._config.get(key, self._default_config[key])

    def __str__(self):
        out = ["%s" % self.__class__.__name__]
        for key in self._default_config.keys():
            out.append("%s: %s" % (key, self.__getattr__(key)))
        return "\n".join(out)


class LoggingConfig(BaseConfig):
    _default_config = {
        'level': 10,
        'filename': '/var/log/yandex/abyssync.log',
        'backup_count': 5,
        'max_bytes': 512 * 1024 * 1024,
    }


class ZookeeperPathsConfig(BaseConfig):
    _default_config = {
        'apps': 'monops/managed-apps',
        'dcinfo': 'closed-dc-info',
        'prefix': '/all_chemodan/',
    }


class ZookeeperConfig(BaseConfig):
    environment_mapping = {
        'ZOOKEEPER_CREDS': 'creds',
    }

    _default_config = {
        'hosts': ["zk01.example.com:2181", "zk02.example.com:2181", "zk03.example.com:2181"],
        'connection_timeout': 5,
        'creds': 'example:example',
        'jaas': None,
        'paths': ZookeeperPathsConfig(),
    }

    @staticmethod
    def load_config(config={}):
        config['paths'] = ZookeeperPathsConfig(config.get('paths', {}))
        return config

    def after_init(self):
        if 'jaas' in self._config:
            try:
                creds = self.parse_jaas_file(self._config['jaas'])
            except Exception as e:
                print(e, file=sys.stderr)
            else:
                self._config['creds'] = creds

    @staticmethod
    def parse_jaas_file(filename):
        if not filename:
            raise FileNotFoundError("empty path")
        conf = {}
        with open(filename, 'r') as jaas:
            for line in jaas:
                if '=' not in line:
                    continue
                key, value = map(lambda x: x.strip(), line.split('=', maxsplit=1))
                conf[key] = value.strip('";')

        return '%s:%s' % (conf['username'], conf['password'])


class CommandsConfig(BaseConfig):
    _default_config = {
        'open': 'echo "Open"',
        'close': 'echo "Close"',
    }


class Config(BaseConfig):
    environment_mapping = {
        'YANDEX_ENVIRONMENT': 'env',
        'DC': 'dc',
        'APPNAME': 'appname',
    }

    _default_config = {
        'appname': 'example',
        'dc': 'none',
        'env': 'testing',
        'disabled_file': '/etc/yandex/abyssync.disabled',
        'sleep': 1,
        'isdbpresent': False,
        'logging': LoggingConfig(),
        'zookeeper': ZookeeperConfig(),
        'commands': CommandsConfig(),
    }

    def __init__(self, config={}):
        super().__init__(config)
        self.__logger = None

    @staticmethod
    def load_config(config={}):
        filename = "/etc/yandex/abyssync.yaml"
        if len(sys.argv) > 1:
            filename = sys.argv[1]
        try:
            config.update(yaml.safe_load(open(filename)))
        except Exception as e:
            print('Unable to load yaml settings, check that config in YAML format: %s' % e, file=sys.stderr)
        config['logging'] = LoggingConfig(config.get('logging', {}))
        config['zookeeper'] = ZookeeperConfig(config.get('zookeeper', {}))
        config['commands'] = CommandsConfig(config.get('commands', {}))
        return config

    def after_init(self):
        self.simplify_env()
        self.simplify_dc()

    def simplify_env(self):
        if 'env' in self._config and (self._config['env'] in ('stable', 'corp-stable')):
            self._config['env'] = 'production'

    def simplify_dc(self):
        if 'dc' in self._config:
            self._config['dc'] = self._config['dc'].lower()

    @property
    def logger(self):
        if self.__logger is None:
            self.__logger = EnvironmentLogger(self.env, self.appname, self.logging)
        return self.__logger
