import logging
import logging.handlers
import socket


def tskv_escape(message):
    return (
        str(message)
        .replace('\\', '\\\\')
        .replace('\t', r'\t')
        .replace('\r', r'\r')
        .replace('\n', r'\n')
        .replace('\0', r'\0')
    )


class EnvironmentLogger(logging.LoggerAdapter):
    def __init__(self, env, appname, logging_config):
        logger = logging.getLogger('abyssync')
        logger.setLevel(logging_config.level if logging_config.level > 0 else 10)

        file_handler = logging.handlers.RotatingFileHandler(
            filename=logging_config.filename,
            mode='a',
            maxBytes=logging_config.max_bytes,
            backupCount=logging_config.backup_count,
        )
        file_handler.setFormatter(
            logging.Formatter(
                "tskv\tappname=%(appname)s\tunixtime=%(created)d\t"
                "timestamp=%(asctime)s\ttimezone=+0300\tpid=%(process)s\tmodule=%(module)s\t"
                "env=%(env)s\tlevel=%(levelname)s\thost=%(host)s\tmessage=%(message)s"
            )
        )
        file_handler.setLevel(logging_config.level if logging_config.level > 0 else 10)
        logger.addHandler(file_handler)

        super(EnvironmentLogger, self).__init__(
            logger,
            {
                'env': env,
                'appname': appname,
                'host': socket.gethostname(),
            },
        )

    def process(self, message, *args, **kwargs):
        return super(EnvironmentLogger, self).process(str(message), *args, **kwargs)
