import os
import signal
import sys
import time

from .commands import execute_command
from .config import Config
from .status import Status
from .zookeeper import Zookeeper
from kazoo.handlers.threading import KazooTimeoutError
from kazoo.exceptions import KazooException


def step(zk, status, config):
    status.message = "OK"
    try:
        if not zk.check_connect():
            config.logger.error("reconnect to zookeeper failed.")
    except (KazooException, KazooTimeoutError) as e:
        config.logger.error("Failed to connect to ZK: %s %s" % (e.__class__, str(e)))
        return status.error(e)

    try:
        dc_info = zk.get_dc_info(config.dc)
    except Exception as e:
        status.is_zk_connected = False
        return status.error(e)

    status.is_zk_connected = True

    if os.path.exists(config.disabled_file):
        return status.disable()

    status.is_disabled = False

    is_app_closed = dc_info.is_app_closed(config.appname, config.isdbpresent)
    if is_app_closed:
        status.message = "Closed by ZK"
        if not status.is_closed:
            try:
                execute_command(config.logger, config.commands.close)
            except Exception as e:
                status.error(e)
            status.is_closed = is_app_closed
            return status
    elif status.is_closed is None or status.is_closed:
        try:
            execute_command(config.logger, config.commands.open)
        except Exception as e:
            return status.error(e)
        else:
            status.is_closed = is_app_closed

    status.code = 0
    return status


def main():
    config = Config()

    def signal_handler(sig, _):
        config.logger.warning("signal %s has caught. Stop" % sig)
        sys.exit(0)

    signal.signal(signal.SIGINT, signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    zk = Zookeeper(config)
    # Connect to ZK, retry forever with config.sleep pause until we are connected.
    while True:
        try:
            zk.start()
            break
        except (KazooException, KazooTimeoutError) as e:
            config.logger.error(e)
            time.sleep(config.sleep)

    zk.set_app_info(config.appname)
    status = Status(message="OK", code=0, is_zk_connected=True, is_disabled=False, logger=config.logger)
    config.logger.info("abyssync started")

    while True:
        status = step(zk, status, config)
        status.log()
        time.sleep(config.sleep)
