import json
import posixpath

from kazoo.client import KazooClient
from kazoo.exceptions import NoNodeError, NodeExistsError
from kazoo.handlers.threading import SequentialThreadingHandler

from .dc_info import DCInfo


class Zookeeper(KazooClient):
    def __init__(self, config):
        super(Zookeeper, self).__init__(
            hosts=config.zookeeper.hosts,
            handler=SequentialThreadingHandler(),
            auth_data=[('sasl', config.zookeeper.creds)],
            timeout=config.zookeeper.connection_timeout,
            logger=config.logger,
        )
        self.chroot = posixpath.join(config.zookeeper.paths.prefix, config.env)
        self.paths = config.zookeeper.paths

    def check_connect(self):
        if not self.connected:
            self.logger.warning("zookeeper disconnected. try again. state:%s" % self.state)
            self._reset()
            self.restart()
            return self.connected
        return True

    def set_app_info(self, appname):
        path = posixpath.join(self.paths.apps, appname)
        data = bytes(json.dumps({'id': appname}), encoding='utf8')
        try:
            return self.create(path, data, makepath=True)
        except NodeExistsError:
            return posixpath.join(self.chroot, path)

    def get_dc_info(self, dc):
        path = posixpath.join(self.paths.dcinfo, dc)
        if not self.exists(path):
            return DCInfo(dc)

        try:
            data = json.loads(self.get(path)[0])
        except NoNodeError:
            return DCInfo(dc)
        else:
            return DCInfo(dc, json=data)
