#!/usr/bin/python

import os
import sys
import time
import argparse
import logging as log

try:
    import json
except ImportError:
    import simplejson as json

import urllib


log.basicConfig(format="%(levelname)s, line %(lineno)3d: %(message)s", level=10)


def touch(path):
    try:
        os.utime(path, None)
    except os.error:
        pass


class PathExpander(argparse.Action):
    def __call__(self, parser, ns, values, option_string=None):
        setattr(ns, self.dest, os.path.expanduser(values))


class ConductorDynamicInventory():

    def __init__(self):
        self.default_conductor_projects = ["disk", "sync", "dataapi"]
        self.default_conductor_api = 'https://c.yandex-team.ru'
        self.default_cache_file = '~/.ansible/dynamic-inventory-cache.json'
        self.cache_live_time = 3600

        self.parse_args()
        self.parse_config()

        if self.args.api_url:
            self.api_url = self.args.api_url
        else:
            self.api_url = self.conf.get('api_url', self.default_conductor_api)

        if self.args.projects:
            self.projects = self.args.projects
        else:
            self.projects = ','.join(self.conf.get('projects', self.default_conductor_projects))

        if self.args.cache_file:
            self.cache_file = self.args.cache_file
        else:
            self.cache_file = os.path.expanduser(self.default_cache_file)

        if not os.path.exists(os.path.dirname(self.cache_file)):
            os.mkdir(os.path.dirname(self.cache_file))

        mode = 'r' if os.path.exists(self.cache_file) else 'w'
        self.cache_file = open(self.cache_file, mode)

        self.run()

    def parse_args(self):
        parser = argparse.ArgumentParser(description='Produce an Ansible Inventory'
                                         ' file based on Conductor API info')

        parser.add_argument('--config-file', '-f', type=str,
                            default=os.path.expanduser(
                                '~/.ansible/dynamic-inventory-config.json'),
                            help='Use specified config file, must be a valid JSON'
                            ' file (default: ~/.ansible/dynamic-inventory-config.json)',
                            action=PathExpander)
        parser.add_argument('--projects', '-p', type=str, default=None,
                            help='Specify list of a projects to override config file,'
                            ' comma-separated (default: "{}")'.format(', '.join(self.default_conductor_projects)))
        parser.add_argument('--api_url', '-a', type=str, default=None,
                            help='Override config URL where Conductor API is available '
                            '(default: "{}")'.format(self.default_conductor_api))
        parser.add_argument('--cache_file', '-c', type=str, default=None,
                            help='File with cached hosts info from Conductor '
                            '(default: "{}")'.format(self.default_cache_file))
        parser.add_argument('--update', '-u', action='store_true', default=False,
                            help="Force using most actual info, may be slower that"
                            " Conductor's cache (default: False)")

        # Completion args:
        parser.add_argument('--bash-completion-groups', '--bcg',
                            action='store_true', default=False,
                            help='Bash completion mode: print out list of all groups')
        parser.add_argument('--bash-completion-hosts', '--bch',
                            action='store_true', default=False,
                            help='Bash completion mode: print out list of all hosts')
        parser.add_argument('--bash-completion', '--bc',
                            action='store_true', default=False,
                            help='Bash completion mode: print out list of all hosts')

        parser.add_argument('--list', '-l', action='store_true', default=True,
                            help='Dummy flag for Ansible compability (default: True)')
        self.args = parser.parse_args()

    def parse_config(self):
        try:
            self.conf = json.load(open(self.args.config_file, 'r'))
        except Exception as e:
            self.conf = {}
            log.warn("Failed to parse config. {}".format(e))

            if not self.args.api_url:
                self.api_url = self.default_conductor_api

            if not self.args.projects:
                self.projects = ', '.join(self.default_conductor_projects)

    def check_cache(self):
        now = time.time()
        mtime = os.path.getmtime(self.cache_file.name)

        try:
            data = self.cache_file.read()
            data = self.convert_unicode_to_str(data)
        except:
            data = False

        try:
            is_actual = (now - mtime) < self.cache_live_time
        except:
            is_actual = False

        return is_actual, data

    def convert_unicode_to_str(self, input):
        if isinstance(input, dict):
            return {self.convert_unicode_to_str(key): self.convert_unicode_to_str(value)
                    for key, value in input.iteritems()}
        elif isinstance(input, list):
            return [self.convert_unicode_to_str(element) for element in input]
        elif isinstance(input, unicode):
            return input.encode('utf-8')
        else:
            return input

    def save_cache(self, data):
        self.cache_file = open(self.cache_file.name, 'w')

        return self.cache_file.write(data)

    def get_data(self, url):
        response = urllib.urlopen(url)
        if response.getcode() == 200:
            return response.read()
        else:
            return False

    def run(self):
        log.info("Gathering info for projects: %s", self.projects)

        cache_is_actual, cache = self.check_cache()

        if cache_is_actual and cache and not self.args.update:
            log.info("Using local cache")
            data = cache
        else:
            log.info("Local cache is outdated, invalid or force using most actual info")
            log.info("Try use Conductor's data")
            api_cache = '/api/'
            if self.args.update:
                log.info("Force using most actual info")
                api_cache = '/api/'
            self.api_url = self.api_url.rstrip('/') + api_cache

            url = self.api_url + 'generator/rivik.ansible-inventory?projects=' + self.projects

            try:
                remote = self.get_data(url)
            except Exception as e:
                log.warn("Failed get data from {}. {}".format(url, e))

            try:
                json.loads(remote)
                data = remote
            except:
                if cache:
                    log.warn("Remote data is invalid, falling back to local cache")
                    data = cache
                else:
                    log.error("Remote data is invalid and no actual cache present")
                    self.cache_file.close()
                    os.unlink(self.cache_file.name)
                    sys.exit(1)

            if data != cache:
                log.info("Refreshing local cache")
                self.save_cache(data)
            else:
                touch(self.cache_file.name)

        self.report(data)
        sys.exit(0)

    def report(self, data):

        if any((self.args.bash_completion_groups,
                self.args.bash_completion_hosts,
                self.args.bash_completion)):
            report = []

            data = json.loads(data)

            groups_list = filter(lambda g: g != '_meta', data.iterkeys())

            if self.args.bash_completion_groups or self.args.bash_completion:
                report += groups_list

            if self.args.bash_completion_hosts or self.args.bash_completion:
                hosts_list = []
                for g in groups_list:
                    hosts_list += data[g]['hosts']

                hosts_list = list(set(hosts_list))
                report += hosts_list

            report = ' '.join(report)
        else:
            report = data

        print report

if __name__ == '__main__':
    ConductorDynamicInventory()