#!/usr/bin/python

# (c) 2015, Andrey Trubachev <d3rp@yandex-team.ru>
#
# This file is part of Ansible,
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.

DOCUMENTATION = '''
---
module: site_facts
version_added: "2.0"
short_description: get machine facts from Conductor, Bot
description:
  - Get machine facts from Conductor, Bot
options:
  host:
    description:
      - fqdn
    required: true
    default: None

author: Andrey Trubachev <d3rp@yandex-team.ru>
requirements: [ ]
'''

EXAMPLES = '''
# get facts
site_facts: host={{ inventory_hostname }}
'''

import socket
from re import search
from json import loads
from collections import defaultdict


def _fetch_json(module, url):
    """Fetch data (suppose it is json) via http and return dict"""
    response, info = fetch_url(module, url)
    if info['status'] == 200:
        try:
            return loads(response.read())
        except ValueError as e:
            module.fail_json(msg="Cannot load json. Error: {error}".format(error=str(e)))
    else:
        module.fail_json(msg="Failed fetch data from {url}. Response code is {response_code}".format(
            url=url,
            response_code=info['status'])
        )


def _get_inventory_number(module, fqdn):
    """Get machine inventory number from bot.yandex-team.ru"""
    url = 'https://bot.yandex-team.ru/api/osinfo.php?fqdn={fqdn}&output=instance_number'.format(fqdn=fqdn)
    json = _fetch_json(module, url)

    if json['os'] and 'instance_number' in json['os'][0]:
        return json['os'][0]['instance_number']
    else:
        return ''


def _get_cloud_region(fqdn):
    """Get region from FQDN (suppose all FQDN follows naming convention)"""
    cloud_region_map = {
        'f': 'myt5',
        'h': 'sas',
        'i': 'sas',
        'e': 'iva6',
        'g': 'fola'
    }

    # get 'gt' from 'csfront01gt.yandex.ru'
    try:
        code = search('[a-z]+\d+([a-z]+)\.', fqdn).group(1)
        return cloud_region_map[code[0]]
    except:
        return ''


def _get_conductor_host_facts(module, fqdn):
    """Get host facts from conductor"""
    url = 'https://c.yandex-team.ru/api/hosts/{fqdn}?format=json'.format(fqdn=fqdn)
    json = _fetch_json(module, url)

    return json[0]


def _get_tags(module, fqdn):
    """Get tags from conductor"""
    url = 'https://c.yandex-team.ru/api/get_host_tags/{fqdn}?format=json'.format(fqdn=fqdn)
    json = _fetch_json(module, url)

    return json

def main():
    module = AnsibleModule(
        argument_spec=dict(
            host=dict(required=True, type='str'),
            smart=dict(required=False, default=False, type='bool'),
            validate_certs=dict(required=False, default=False, type='bool')
        )
    )

    params = module.params
    host = params['host']

    Tree = lambda: defaultdict(Tree)
    results = Tree()

    results['inventory_number'] = _get_inventory_number(module, host)

    # if inventory_number is empty it means we operate with virtual machine
    if results['inventory_number'] == '':
        results['cloud']['region'] = _get_cloud_region(host)

    for k, v in _get_conductor_host_facts(module, host).items():
        results[k] = v

    #results['fastbone_fqdn'] = _get_fastbone(host, results['short_name'])

    # fill in cloud and eine
    for raw_tag in _get_tags(module, host):
        raw_list = raw_tag.split(':')
        # we process only tag that matches to pattern <namespace>:<key>:<value>
        if len(raw_list) == 3:
            tag_namespace, tag_key, tag_value = raw_list
            if tag_namespace == 'cloud':
                # we hope nic name is always in uppercase on cloud.yandex-team.ru
                if tag_key == 'nic':
                    tag_value = str(tag_value).upper()
            results[tag_namespace][tag_key] = tag_value


    module.exit_json(ansible_facts=results)

# import module snippets
from ansible.module_utils.basic import *
from ansible.module_utils.urls import *

if __name__ == '__main__':
    main()
