# -*- coding: utf-8 -*-

import argparse

import logging
logging.basicConfig(level=logging.INFO, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")
logger = logging.getLogger(__name__)

import os
import sys
sys.path.append(os.getcwd())

from lib.collectors.extractors import HwExtractor, PGExtractor, SharpeiExtractor, QloudExtractor, MDBExtractor, DeployExtractor
from lib.util import date_to_timestamp, yesterday
import datetime


EXTRACTORS = {'hw': HwExtractor,
              'pg': PGExtractor,
              'sharpei': SharpeiExtractor,
              'qloud': QloudExtractor,
              'mdb': MDBExtractor,
              'deploy': DeployExtractor}


def extract(date, env):
    exc = EXTRACTORS.get(env)()
    if not exc:
        logger.warning('Exctractor for %s environment is not defined.' % env)
        return
    # parsed and collected inside
    exc.process(date)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Collect capacity data and upload to stat')
    parser.add_argument('-s', '--start', type=str, default=yesterday(), help='Start date; Yesterday by default')
    parser.add_argument('-d', '--day', type=int, help='Collect N days from start date', default=1)
    parser.add_argument('-e', '--env', nargs='+', required=True, help='Collected environments, "all" or any of: %s' %
                                                                      ', '.join(EXTRACTORS.keys()), )
    parser.add_argument('--hosts', nargs='+', help='Collect for specific hosts (with 1 env only)')
    args = parser.parse_args()
    if 'all' in args.env:
        args.env = list(EXTRACTORS.keys())
    else:
        args.env = [e for e in args.env if e in EXTRACTORS]
    args.start = date_to_timestamp(args.start)
    if args.hosts and len(args.env) != 1:
        exit('Select only 1 environment')

    sday = datetime.datetime.fromtimestamp(args.start)
    for day in range(args.day):
        for env in args.env:
            dt = sday - datetime.timedelta(days=day)
            logger.info('Start extracting logs at %s env per %s' % (env, dt))
            extract(dt, env)
