# -*- coding: utf-8 -*-

import itertools
import os
import sys
import time
import logging

from collections import defaultdict
from requests.exceptions import ConnectionError, ConnectTimeout
from urllib import quote

from xmltodict import parse
from lib.util import get_url
from lib.decorators import cached, retry

logging.basicConfig(level=logging.DEBUG, format="[%(asctime)s] %(levelname)s %(name)s %(message)s")
logger = logging.getLogger(__name__)


def urler(*args):
    return "/".join(str(url) for url in args)


class Conductor:
    def __init__(self):
        self.base_url = 'https://c.yandex-team.ru/api'
        self.token = os.getenv("CONDUCTOR_TOKEN")
        if not self.token:
            raise EnvironmentError("$CONDUCTOR_TOKEN is not set")
        self.oauth_header = {"Authorization": "OAuth %s" % self.token}

    def get_projects(self):
        url = urler(self.base_url, 'projects')
        return parse(get_url(url, headers=self.oauth_header))

    def get_groups_by_prj(self, project, filter_fn=bool):
        """
        :param project:
        :param filter_fn: filter function
        :return: list of groups
        """
        url = urler(self.base_url, 'projects2groups', project)
        res = get_url(url, headers=self.oauth_header).strip().split('\n')
        return list(filter(filter_fn, res))

    def get_groups_with_hosts_by_prj(self, project, filter_fn=bool, recursive=True, fields=('fqdn', 'id')):
        groups = self.get_all_hosts_by_prj(project, filter_fn)
        res = {}
        for g in groups:
            res[g] = self.get_hosts_by_group(g, recursive, fields)
            time.sleep(0.2)
        return res

    def get_groups(self, groups):
        """

        :param groups: str or iterable
        :return:
        """
        if isinstance(groups, (str, basestring)):
            url = urler(self.base_url, 'groups', quote(groups))
        else:
            groups = quote(','.join(groups))
            url = urler(self.base_url, 'groups', groups)
        if len(url) >= 256:
            logger.warning('Url is too long: %s %s' % (len(url), url))

        logger.info('Requesting groups: %s' % url)
        cgroups = parse(get_url(url, headers=self.oauth_header))['data']['item']
        if not isinstance(cgroups, list):
            cgroups = [cgroups]

        for i, g in enumerate(cgroups):
            print(i, g)
            for key in ('parents', 'children'):
                raw = (g.get(key) or {}).get('item', [])
                if not isinstance(raw, list):
                    raw = [raw]
                cgroups[i][key] = raw
        return cgroups

    @cached(hours=999)
    @retry(5, 1, 2)
    def get_hosts_by_group(self, group, recursive=False, fields=('fqdn', 'id')):
        available_fields = ('id', 'fqdn', 'short_name', 'description', 'created_at',
                            'updated_at', 'datacenter_name', 'root_datacenter_name')
        if any([h for h in fields if h not in available_fields]):
            raise KeyError('Invalid fields requested')
        url = urler(self.base_url, 'groups2hosts', group)
        if not recursive:
            url += '?recursive=no'
        res = get_url(url, headers=self.oauth_header)
        if res is None:
            raise ValueError("Invalid group passed: %s" % group)
        return [x.strip() for x in filter(bool, res.strip().split('\n'))]

    @cached(hours=999)
    def get_all_hosts_by_prj(self, prj, filter_fn=bool, sleep=0.1):
        hosts = defaultdict(list)
        for group in self.get_groups_by_prj(prj, filter_fn):
            try:
                hosts[group] = self.get_hosts_by_group(group)
                time.sleep(sleep)
            except (ConnectionError, ConnectTimeout):
                time.sleep(1)
                hosts[group] = self.get_hosts_by_group(group)
        return hosts

    # def build_prj_tree(self, prj):
    #     url = urler(self.base_url, 'groups', prj)
    #     groups = parse(get_url(url, headers=self.oauth_header))

    # def get_tree(self, prj, force=False):
    #     if (not getattr(self, '_tree', None)) or force:
    #         self._tree = self.build_prj_tree(prj)
    #     return self._tree

conductor = Conductor()
